{ *************************************************************************** }
{                                                                             }
{ Kylix and Delphi Cross-Platform Visual Component Library                    }
{                                                                             }
{ Copyright (c) 1999, 2005 Borland Software Corporation                       }
{                                                                             }
{ *************************************************************************** }


unit SqlExpr;

{$R-,T-,H+,X+}

interface

{$IFDEF MSWINDOWS}
uses Windows, SysUtils, Variants, Classes, DB, DBCommon, DBCommonTypes,
  DBXpress, SqlTimSt, WideStrings;
{$ENDIF}
{$IFDEF LINUX}
uses Libc, SysUtils, Variants, Classes, DB, DBCommon, DBXpress, SqlTimSt;
{$ENDIF}

const

  SSelect         =   'select';               { Do not localize }
  SSelectStar     =   ' select * ';           { Do not localize }
  SSelectStarFrom =   ' select * from ';      { Do not localize }
  SSelectSpaces   =   ' select ';             { Do not localize }
  SWhere          =   ' where ';              { Do not localize }
  SAnd            =   ' and ';                { Do not localize }
  SOrderBy        =   ' order by ';           { Do not localize }
  SParam          =   '?';                    { Do not localize }
  DefaultCursor   =   0;
  HourGlassCursor =   -11;

{ Default Max BlobSize }

  DefaultMaxBlobSize = -1;   // values are in K; -1 means retrieve actual size

{ Default RowsetSize }

  DefaultRowsetSize = 20;

  TErrorMessageSize = 2048;

{ FieldType Mappings }

  FldTypeMap: TFieldMap = (
    fldUNKNOWN, fldZSTRING, fldINT16, fldINT32, fldUINT16, fldBOOL, // 0..5
    fldFLOAT, fldFLOAT, fldBCD, fldDATE, fldTIME, fldTIMESTAMP, fldBYTES, // 6..12
    fldVARBYTES, fldINT32, fldBLOB, fldBLOB, fldBLOB, fldBLOB, fldBLOB, // 13..19
	  fldBLOB, fldBLOB, fldCURSOR, fldZSTRING, fldWIDESTRING, fldINT64, fldADT, // 20..26
    fldArray, fldREF, fldTABLE, fldBLOB, fldBLOB, fldUNKNOWN, fldUNKNOWN, // 27..33
    fldUNKNOWN, fldZSTRING, fldDATETIME, fldBCD, // 33..37
    fldWIDESTRING, fldBLOB, fldDATETIME, fldZSTRING); // 38..41

  FldSubTypeMap: array[TFieldType] of Word = (
    0, 0, 0, 0, 0, 0, 0, fldstMONEY, 0, 0, 0, 0, 0, 0, fldstAUTOINC, // 0..14
    fldstBINARY, fldstMEMO, fldstGRAPHIC, fldstFMTMEMO, fldstOLEOBJ, // 15..19
    fldstDBSOLEOBJ, fldstTYPEDBINARY, 0, fldstFIXED, 0, // 20..24
    0, 0, 0, 0, 0, fldstHBINARY, fldstHMEMO, 0, 0, 0, 0, 0, 0, // 24..37
    fldstFIXED, fldstWIDEMEMO, fldstORATIMESTAMP, fldstORAINTERVAL); // 38 ..41
  DataTypeMap: array[0..MAXLOGFLDTYPES - 1] of TFieldType = (
    ftUnknown, ftString, ftDate, ftBlob, ftBoolean, ftSmallint,
    ftInteger, ftFloat, ftBCD, ftBytes, ftTime, ftDateTime,
    ftWord, ftInteger, ftUnknown, ftVarBytes, ftUnknown, ftCursor,
    ftLargeInt, ftLargeInt, ftADT, ftArray, ftReference, ftDataSet,
    ftTimeStamp, ftFMTBCD, ftWideString);

  BlobTypeMap: array[fldstMEMO..fldstBFILE] of TFieldType = (
    ftMemo, ftBlob, ftFmtMemo, ftParadoxOle, ftGraphic, ftDBaseOle,
    ftTypedBinary, ftBlob, ftBlob, ftBlob, ftWideMemo, ftOraClob, ftOraBlob,
    ftBlob, ftBlob);

type

{ Forward declarations }

  TSQLConnection = class;
  TCustomSQLDataSet = class;
  TSQLDataSet = class;
  TSQLQuery = class;
  TSQLStoredProc = class;
  TSQLTable = class;

  TLocaleCode = Integer;

  TSQLExceptionType = (exceptConnection, exceptCommand, exceptCursor, exceptMetaData, exceptUseLast);

  PSPParamDesc = ^SPParamDesc;
  SPParamDesc = packed record           { Stored Proc Descriptor }
    iParamNum       : Word;             { Field number (1..n) }
    szName          : WideString;       { Field name }
    iArgType        : TParamType;       { Field type }
    iDataType       : TFieldType;       { Field type }
    iUnits1         : SmallInt;         { Number of Chars, digits etc }
    iUnits2         : SmallInt;         { Decimal places etc. }
    iLen            : LongWord;         { Length in bytes  }
  end;
  SQLSPParamDesc = SPParamDesc;

{ TSQLBlobStream }

  TSQLBlobStream = class(TMemoryStream)
  private
    FDataSet: TCustomSQLDataSet;
    FField: TBlobField;
    FFieldNo: Integer;
  public
    constructor Create(Field: TBlobField; Mode: TBlobStreamMode = bmRead);
    destructor Destroy; override;
    procedure ReadBlobData;
  end;

  TConnectionUserType = (eUserMonitor, eUserDataSet);

{ Forward declear }

  TISQLConnection = class;
  TISQLCommand = class;
  TISQLCursor = class;
  TISQLMetaData = class;
  TFLDDesc = class;

  TFLDDescRef = class of TFLDDesc;

{ ISQLConnection wrapper class }

  TISQLConnection = class
  public
    constructor Create(NewConnection: ISQLConnection); virtual; abstract;
    function connect(): SQLResult; overload; virtual; abstract;
    function connect(ServerName: PWideChar; UserName: PWideChar;
                          Password: PWideChar): SQLResult; overload; virtual; abstract;
    function disconnect: SQLResult; virtual; abstract;
    function getSQLCommand(var pComm: TISQLCommand): SQLResult; virtual; abstract;
    function getSQLMetaData(var pMetaData: TISQLMetaData): SQLResult; virtual; abstract;
    function SetOption(eConnectOption: TSQLConnectionOption;
            lValue: LongInt): SQLResult; virtual; abstract;
    function SetStringOption(eConnectOption: TSQLConnectionOption;
            const lValue: WideString): SQLResult; virtual; abstract;
    function GetOption(eDOption: TSQLConnectionOption; PropValue: Pointer;
            MaxLength: SmallInt; out Length: SmallInt): SQLResult; virtual; abstract;
    function GetStringOption(eDOption: TSQLConnectionOption;
            var str: WideString): SQLResult; virtual; abstract;
    function beginTransaction(TranID: LongWord): SQLResult; virtual; abstract;
    function commit(TranID: LongWord): SQLResult; virtual; abstract;
    function rollback(TranID: LongWord): SQLResult; virtual; abstract;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; virtual; abstract;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; virtual; abstract;
    function getErrorMessage(var Error: WideString): SQLResult; overload; virtual; abstract;
    function getFldDescClass: TFldDescRef; virtual; abstract;
  end;

{ ISQLCommand wrapper class }

  TISQLCommand = class
  private
    FSQLCursor: TISQLCursor;
  public
    destructor Destroy; override;
    function SetOption(
      eSqlCommandOption: TSQLCommandOption;
      ulValue: Integer): SQLResult; virtual; abstract;
    function SetStringOption(
      eSqlCommandOption: TSQLCommandOption;
      const ulValue: WideString): SQLResult; virtual; abstract;
    function GetOption(eSqlCommandOption: TSQLCommandOption;
      PropValue: Pointer;
      MaxLength: SmallInt; out Length: SmallInt): SQLResult; virtual; abstract;
    function GetStringOption(eSqlCommandOption: TSQLCommandOption;
      var ulValue: WideString): SQLResult; virtual; abstract;
    function setParameter(
      ulParameter: Word ;
      ulChildPos: Word ;
      eParamType: TSTMTParamType ;
      uLogType: Word;
      uSubType: Word;
      iPrecision: Integer;
      iScale: Integer;
      Length: LongWord ;
      pBuffer: Pointer;
      lInd: Integer): SQLResult; virtual; abstract;
    function getParameter(ParameterNumber: Word; ulChildPos: Word; Value: Pointer;
      Length: Integer; var IsBlank: Integer): SQLResult; virtual; abstract;
    function prepare(SQL: PWideChar; ParamCount: Word): SQLResult; virtual; abstract;
    function execute(var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function executeImmediate(SQL: PWideChar; var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getNextCursor(var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getRowsAffected(var Rows: LongWord): SQLResult; virtual; abstract;
    function close: SQLResult; virtual; abstract;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; virtual; abstract;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; virtual; abstract;
    function getErrorMessage(var Error: WideString): SQLResult; overload; virtual; abstract;
  end;

{ ISQLCursor wrapper class }

  TISQLCursor = class
    function SetOption(eOption: TSQLCursorOption;
                     PropValue: LongInt): SQLResult; virtual; abstract;
    function SetStringOption(eOption: TSQLCursorOption;
                     const str: WideString): SQLResult; virtual; abstract;
    function GetOption(eOption: TSQLCursorOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; virtual; abstract;
    function GetStringOption(eOption: TSQLCursorOption;
                     var str: WideString): SQLResult; virtual; abstract;
    function getCurObjectTypeName(const iFldNum: Word): WideString; virtual; abstract;
    function getErrorMessage(Error: PWideChar): SQLResult; overload;  virtual; abstract;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;  virtual; abstract;
    function getErrorMessage(var Error: WideString): SQLResult; overload; virtual; abstract;
    function getColumnCount(var pColumns: Word): SQLResult;  virtual; abstract;
    function getColumnNameLength(
      ColumnNumber: Word;
      var pLen: Word): SQLResult;  virtual; abstract;
    function getColumnName(ColumnNumber: Word; pColumnName: PWideChar): SQLResult; overload; virtual; abstract;
    function getColumnName(ColumnNumber: Word): WideString; overload; virtual; abstract;
    function getColumnType(ColumnNumber: Word; var puType: Word;
      var puSubType: Word): SQLResult;  virtual; abstract;
    function  getColumnLength(ColumnNumber: Word; var pLength: LongWord): SQLResult;  virtual; abstract;
    function getColumnPrecision(ColumnNumber: Word;
      var piPrecision: SmallInt): SQLResult;  virtual; abstract;
    function getColumnScale(ColumnNumber: Word; var piScale: SmallInt): SQLResult;  virtual; abstract;
    function isNullable(ColumnNumber: Word; var Nullable: LongBool): SQLResult;  virtual; abstract;
    function isAutoIncrement(ColumnNumber: Word; var AutoIncr: LongBool): SQLResult;  virtual; abstract;
    function isReadOnly(ColumnNumber: Word; var ReadOnly: LongBool): SQLResult; virtual; abstract;
    function isSearchable(ColumnNumber: Word; var Searchable: LongBool): SQLResult; virtual; abstract;
    function isBlobSizeExact(ColumnNumber: Word; var IsExact: LongBool): SQLResult; virtual; abstract;
    function next: SQLResult; virtual; abstract;
    function getString(ColumnNumber: Word; Value: PChar;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getWideString(ColumnNumber: Word; Value: PWideChar;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getShort(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getLong(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getInt64(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getDouble(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getBcd(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getTimeStamp(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getTime(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getDate(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getBytes(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getBlobSize(ColumnNumber: Word; var Length: LongWord;
      var IsBlank: LongBool): SQLResult; virtual; abstract;
    function getBlob(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool; Length: LongWord): SQLResult; virtual; abstract;
end;

{ ISQLMetaData wrapper class }

  TISQLMetaData = class
    function SetOption(eDOption: TSQLMetaDataOption;
                     PropValue: LongInt): SQLResult; virtual; abstract;
    function SetStringOption(eDOption: TSQLMetaDataOption;
                     const str: WideString): SQLResult; virtual; abstract;
    function GetOption(eDOption: TSQLMetaDataOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; virtual; abstract;
    function GetStringOption(eDOption: TSQLMetaDataOption;
                     var str: WideString): SQLResult; virtual; abstract;
    function getObjectList(eObjType: TSQLObjectType; var Cursor: TISQLCursor):
                     SQLResult; virtual; abstract;
    function getTables(TableName: PWideChar; TableType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getProcedures(ProcedureName: PWideChar; ProcType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getColumns(TableName: PWideChar; ColumnName: PWideChar;
                     ColType: LongWord; var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getProcedureParams(ProcName: PWideChar; ParamName: PWideChar;
                     var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getIndices(TableName: PWideChar; IndexType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; virtual; abstract;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; virtual; abstract;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; virtual; abstract;
    function getErrorMessage(var Error: WideString): SQLResult; overload; virtual; abstract;
  end;

{ TSQLMonitor }

  pSQLTRACEDesc30 = ^SQLTRACEDesc30;
  SQLTRACEDesc30 = packed record             { trace callback info }
    pszTrace        : array [0..1023] of WideChar;
    eTraceCat       : TRACECat;
    ClientData      : Integer;
    uTotalMsgLen    : Word;
  end;

  pSQLTRACEDesc25 = ^SQLTRACEDesc25;
  SQLTRACEDesc25 = packed record             { trace callback info }
    pszTrace        : array [0..1023] of AnsiChar;
    eTraceCat       : TRACECat;
    ClientData      : Integer;
    uTotalMsgLen    : Word;
  end;

  TTraceEvent30 = procedure(Sender: TObject; CBInfo: pSQLTRACEDesc30; var LogTrace: Boolean) of object;
  TTraceLogEvent30 = procedure(Sender: TObject; CBInfo: pSQLTRACEDesc30) of object;
  TTraceEvent25 = procedure(Sender: TObject; CBInfo: pSQLTRACEDesc25; var LogTrace: Boolean) of object;
  TTraceLogEvent25 = procedure(Sender: TObject; CBInfo: pSQLTRACEDesc25) of object;

  pSQLTRACEDesc = pSQLTRACEDesc30;
  SQLTRACEDesc = SQLTRACEDesc30;
  TTraceEvent = TTraceEvent30;
  TTraceLogEvent = TTraceLogEvent30;

  TSQLMonitor = class(TComponent)
  private
    FActive: Boolean;
    FAutoSave: Boolean;
    FFileName: string;
    FKeepConnection: Boolean;
    FMaxTraceCount: Integer;
    FOnTrace: TTraceEvent;
    FOnLogTrace: TTraceLogEvent;
    FSQLConnection: TSQLConnection;
    FStreamedActive: Boolean;
    FTraceFlags: TSQLTraceFlags;
    FTraceList: TWideStrings;
    procedure CheckInactive;
    function GetTraceCount: Integer;
  protected
    function InvokeCallBack(CallType: TRACECat; CBInfo: Pointer): CBRType; stdcall;
    procedure SetActive(Value: Boolean);
    procedure SetSQLConnection(Value: TSQLConnection);
    procedure SetStreamedActive;
    procedure SetTraceList(Value: TWideStrings);
    procedure SetFileName(const Value: String);
    procedure SwitchConnection(const Value: TSQLConnection);
    procedure Trace(Desc: pSQLTraceDesc; LogTrace: Boolean); virtual;
    procedure UpdateTraceCallBack;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure LoadFromFile(AFileName: string);
    procedure SaveToFile(AFileName: string);
    property MaxTraceCount: Integer read FMaxTraceCount write FMaxTraceCount;
    property TraceCount: Integer read GetTraceCount;
  published
    property Active: Boolean read FActive write SetActive default False;
    property AutoSave: Boolean read FAutoSave write FAutoSave default False;
    property FileName: string read FFileName write SetFileName;
    property OnLogTrace: TTraceLogEvent read FOnLogTrace write FOnLogTrace;
    property OnTrace: TTraceEvent read FOnTrace write FOnTrace;
{   property TraceFlags not supported in DBExpress 1.0 }
    property TraceList: TWideStrings read FTraceList write SetTraceList stored False;
    property SQLConnection: TSQLConnection read FSQLConnection write SetSQLConnection;
  end;

{ TSQLConnection }

  TLocale = Pointer;

  EConnectFlag = (eConnect, eReconnect, eDisconnect);

  TSchemaType = (stNoSchema, stTables, stSysTables, stProcedures, stColumns,
    stProcedureParams, stIndexes, stPackages, stUserNames);

  TConnectionState = (csStateClosed, csStateOpen, csStateConnecting,
    csStateExecuting, csStateFetching, csStateDisconnecting);

  TTableScope = (tsSynonym, tsSysTable, tsTable, tsView);

  TTableScopes = set of TTableScope;

  TConnectChangeEvent = procedure(Sender: TObject; Connecting: Boolean) of object;

  TSQLConnectionLoginEvent = procedure(Database: TSQLConnection;
    LoginParams: TWideStrings) of object;

  TSQLConnection = class(TCustomConnection)
  private
    FSelectStatements: LongWord;
    FPrevSelectStatements: LongWord;
    FActiveStatements: LongWord;
    FAutoClone: Boolean;
    FCloneParent: TSQLConnection;
    FConnectionState: TConnectionState;
    FConnectionName: string;
    FConnectionRegistryFile: string;
    FDriverName: string;
    FDriverRegistryFile: string;
    FGetDriverFunc: string;
    FTransactionCount: Integer;
    FIsCloned: Boolean;
    FISQLConnection: TISQLConnection;
    FKeepConnection: Boolean;
    FLastError: string;  // DBExpress GetError() clears error; need to save last
    FLibraryName: string;
    FLoadParamsOnConnect: Boolean;
    FMonitorUsers: TList;
    FOnLogin: TSQLConnectionLoginEvent;
    FParams: TWideStrings;
    FParamsLoaded: Boolean;
    FMaxStmtsPerConn: LongWord;
    FQuoteChar: WideString;
    FDefaultSchemaName: WideString;
    FRefCount: Integer;
    FSQLDllHandle: THandle;
    FSQLDriver: ISQLDriver;
    FSQLHourGlass: Boolean;
    FSQLMetaData: TISQLMetaData;
    FSupportsMultiTrans: LongBool;
    FTableScope: TTableScopes;
    FTraceCallbackEvent: TSQLCallbackEvent;
    FTraceClientData: Integer;
    FTransactionsSupported: LongBool;
    FVendorLib: string;
    FTransIsoLevel: TTransIsolationLevel;
    FLoginUsername: WideString;
    procedure CheckActive;
    procedure CheckInactive;
    procedure CheckLoginParams;
    procedure ClearConnectionUsers;
    procedure ClearMonitors;
    procedure FreeSchemaTable(DataSet: TCustomSQLDataSet);
    function GetConnectionForStatement: TSQLConnection;
    function GetConnectionName: string;
    function GetFDriverRegistryFile: string;
    function GetLocaleCode: TLocaleCode;
    function GetInTransaction: Boolean;
    function GetLibraryName: string;
    procedure GetLoginParams(LoginParams: TWideStrings);
    function GetQuoteChar: WideString;
    function GetVendorLib: string;
    procedure Login(LoginParams: TWideStrings);
    function OpenSchemaTable(eKind: TSchemaType; SInfo: WideString; SQualifier: WideString = ''; SPackage: WideString = ''): TCustomSQLDataSet;overload;
    function OpenSchemaTable(eKind: TSchemaType; SInfo: WideString; SQualifier: WideString = ''; SPackage: WideString = ''; SSchemaName: WideString = ''): TCustomSQLDataSet;overload;
    procedure RegisterTraceMonitor(Client: TObject);
    procedure RegisterTraceCallback(Value: Boolean);
    procedure SetConnectionParams;
    procedure SetConnectionName(Value: string);
    procedure SetDriverName(Value: string);
    procedure SetKeepConnection(Value: Boolean);
    procedure SetParams(Value: TWideStrings);
    procedure SetCursor(CursorType: Integer);
    procedure SetLocaleCode(Value: TLocaleCode);
//    function SQLTraceCallback(CBInfo: Pointer): CBRType;
    procedure UnregisterTraceMonitor(Client: TObject);
  protected
    function Check(status: SQLResult): SQLResult;
    procedure CheckConnection(eFlag: eConnectFlag);
    procedure CheckDisconnect; virtual;
    procedure ConnectionOptions; virtual;
    procedure DoConnect; override;
    procedure DoDisconnect; override;
    function GetConnected: Boolean; override;
    function GetDataSet(Index: Integer): TCustomSQLDataSet; reintroduce;
    procedure Loaded; override;
    procedure LoadSQLDll;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure OpenSchema(eKind: TSchemaType; sInfo: Widestring; List: TWideStrings); overload;
    procedure OpenSchema(eKind: TSchemaType; sInfo, SSchemaName: WideString; List: TWideStrings); overload;
    procedure SQLError(OpStatus: SQLResult; eType: TSQLExceptionType; const Command: TISQLCommand = nil);
    property Connection: TISQLConnection read FISQLConnection;
    property ConnectionRegistryFile: string read FConnectionRegistryFile;
    property Driver: ISQLDriver read FSQLDriver;
    property DriverRegistryFile: string read GetFDriverRegistryFile;
    property LastError: string read FLastError write FLastError;
    property QuoteChar: WideString read FQuoteChar;
    property SQLDllHandle: THandle read FSQLDllHandle write FSQlDllHandle;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CloneConnection: TSQLConnection;
    procedure CloseDataSets;
    procedure Commit( TransDesc: TTransactionDesc);
    function Execute(const SQL: WideString; Params: TParams;
      ResultSet: Pointer = nil): Integer;
    function ExecuteDirect(const SQL: WideString): Integer;
    procedure GetFieldNames(const TableName: string; List: TStrings); overload; deprecated;
    procedure GetFieldNames(const TableName: string; SchemaName: string; List: TStrings); overload; deprecated;
    procedure GetFieldNames(const TableName: WideString; List: TWideStrings); overload;
    procedure GetFieldNames(const TableName: WideString; SchemaName: WideString; List: TWideStrings); overload;
    procedure GetIndexNames(const TableName: string; List: TStrings); overload; deprecated;
    procedure GetIndexNames(const TableName, SchemaName: string; List: TStrings); overload; deprecated;
    procedure GetIndexNames(const TableName: WideString; List: TWideStrings); overload;
    procedure GetIndexNames(const TableName, SchemaName: WideString; List: TWideStrings); overload;
    procedure GetProcedureNames(List: TStrings); overload; deprecated;
    procedure GetProcedureNames(const PackageName: string; List: TStrings); overload; deprecated;
    procedure GetProcedureNames(const PackageName, SchemaName: string; List: TStrings); overload; deprecated;
    procedure GetProcedureNames(List: TWideStrings); overload;
    procedure GetProcedureNames(const PackageName: Widestring; List: TWideStrings); overload;
    procedure GetProcedureNames(const PackageName, SchemaName: WideString; List: TWideStrings); overload;
    procedure GetPackageNames(List: TStrings); overload; deprecated;
    procedure GetPackageNames(List: TWideStrings); overload;
    procedure GetSchemaNames(List: TStrings); overload;
    procedure GetSchemaNames(List: TWideStrings); overload;
    function GetDefaultSchemaName: WideString;
    procedure GetProcedureParams(ProcedureName : WideString; List: TList); overload;
    procedure GetProcedureParams(ProcedureName, PackageName: WideString; List: TList); overload;
    procedure GetProcedureParams(ProcedureName, PackageName, SchemaName: Widestring; List: TList); overload;
    procedure GetTableNames(List: TStrings; SystemTables: Boolean = False); overload;
    procedure GetTableNames(List: TStrings; SchemaName: WideString; SystemTables: Boolean = False); overload;
    procedure GetTableNames(List: TWideStrings; SystemTables: Boolean = False); overload;
    procedure GetTableNames(List: TWideStrings; SchemaName: WideString; SystemTables: Boolean = False); overload;
    procedure LoadParamsFromIniFile( FFileName: WideString = '');
    procedure Rollback( TransDesc: TTransactionDesc);
    procedure SetTraceCallbackEvent(Event: TSQLCallbackEvent; IClientInfo: Integer);
    procedure StartTransaction( TransDesc: TTransactionDesc);
    function GetLoginUsername: WideString;
    property ActiveStatements: LongWord read FActiveStatements;
    property AutoClone: Boolean read FAutoClone write FAutoClone default True;
    property ConnectionState: TConnectionState read FConnectionState write FConnectionState;
    property DataSets[Index: Integer]: TCustomSQLDataSet read GetDataSet;
    property InTransaction: Boolean read GetInTransaction;
    property LocaleCode: TLocaleCode read GetLocaleCode write SetLocaleCode default TLocaleCode(0);
    property MaxStmtsPerConn: LongWord read FMaxStmtsPerConn;
    property MetaData: TISQLMetaData read FSQLMetaData;
    property MultipleTransactionsSupported: LongBool read FSupportsMultiTrans;
    property ParamsLoaded: Boolean read FParamsLoaded write FParamsLoaded;
    property SQLConnection: TISQLConnection read FISQLConnection write FISQLConnection;
    property SQLHourGlass: Boolean read FSQLHourGlass write FSQLHourGlass default True;
    property TraceCallbackEvent: TSQLCallbackEvent read FTraceCallbackEvent;
    property TransactionsSupported: LongBool read FTransactionsSupported;
//    property Locale: TLocale read FLocale;
  published
    property ConnectionName: string read GetConnectionName write SetConnectionName;
    property DriverName: string read FDriverName write SetDriverName;
    property GetDriverFunc: string read FGetDriverFunc write FGetDriverFunc;
    property KeepConnection: Boolean read FKeepConnection write SetKeepConnection default True;
    property LibraryName: string read GetLibraryName write FLibraryName;
    property LoadParamsOnConnect: Boolean read FLoadParamsOnConnect write FLoadParamsOnConnect default False;
    property LoginPrompt default True;
    property Params: TWideStrings read FParams write SetParams;
    property TableScope: TTableScopes read FTableScope write FTableScope default [tsTable, tsView];
    property VendorLib: string read GetVendorLib write FVendorLib;
    property AfterConnect;
    property AfterDisconnect;
    property BeforeConnect;
    property BeforeDisconnect;
    property OnLogin: TSQLConnectionLoginEvent read FOnLogin write FOnLogin;
    property Connected;
  end;

{ TSQLDataLink }

  TSQLDataLink = class(TDetailDataLink)
  private
    FSQLDataSet: TCustomSQLDataSet;
  protected
    procedure ActiveChanged; override;
    procedure CheckBrowseMode; override;
    function GetDetailDataSet: TDataSet; override;
    procedure RecordChanged(Field: TField); override;
  public
    constructor Create(ADataSet: TCustomSQLDataSet);
  end;

{ FLDDesc wrapper }

  TFLDDesc = class
  protected
    function GetiFldNum: Word; virtual; abstract;
    function GetszName: WideString; virtual; abstract;
    function GetiFldType: Word; virtual; abstract;
    function GetiSubType: Word; virtual; abstract;
    function GetiUnits1: SmallInt; virtual; abstract;
    function GetiUnits2: SmallInt; virtual; abstract;
    function GetiOffset: Word; virtual; abstract;
    function GetiLen: LongWord; virtual; abstract;
    function GetiNullOffset: Word; virtual; abstract;
    function GetefldvVchk: FLDVchk; virtual; abstract;
    function GetefldrRights: FLDRights; virtual; abstract;
    function GetbCalcField: WordBool; virtual; abstract;
    procedure SetiFldNum(Value: Word); virtual; abstract;
    procedure SetszName(Value: WideString); virtual; abstract;
    procedure SetiFldType(Value: Word); virtual; abstract;
    procedure SetiSubType(Value: Word); virtual; abstract;
    procedure SetiUnits1(Value: SmallInt); virtual; abstract;
    procedure SetiUnits2(Value: SmallInt); virtual; abstract;
    procedure SetiOffset(Value: Word); virtual; abstract;
    procedure SetiLen(Value: LongWord); virtual; abstract;
    procedure SetiNullOffset(Value: Word); virtual; abstract;
    procedure SetefldvVchk(Value: FLDVchk); virtual; abstract;
    procedure SetefldrRights(Value: FLDRights); virtual; abstract;
    procedure SetbCalcField(Value: WordBool); virtual; abstract;
  public
    property iFldNum: Word read GetiFldNum write SetiFldNum;
    property szName: WideString read GetszName write SetszName;
    property iFldType: Word read GetiFldType write SetiFldType;
    property iSubType: Word read GetiSubType write SetiSubType;
    property iUnits1: SmallInt read GetiUnits1 write SetiUnits1;
    property iUnits2: SmallInt read GetiUnits2 write SetiUnits2;
    property iOffset: Word read GetiOffset write SetiOffset;
    property iLen: LongWord read GetiLen write SetiLen;
    property iNullOffset: Word read GetiNullOffset write SetiNullOffset;
    property efldvVchk: FLDVchk read GetefldvVchk write SetefldvVchk;
    property efldrRights: FLDRights read GetefldrRights write SetefldrRights;
    property bCalcField: WordBool read GetbCalcField write SetbCalcField;
  end;

{ TCustomSQLDataSet }

  TSQLSchemaInfo = record
    FType: TSchemaType;
    ObjectName: WideString;
    Pattern: WideString;
    PackageName : WideString;
  end;

  TFieldDescList = array of TFLDDesc;

  TParseSqlEvent = procedure(var FieldNames: TWideStrings; SQL: WideString;
      var TableName: WideString) of object;
  TParseInsertSqlEvent = procedure(var FieldNames: TWideStrings; SQL: WideString;
      var BindAllFields: Boolean; var TableName: WideString) of object;

  TCustomSQLDataSet = class(TWideDataSet)
  private
    FBlobBuffer: TBlobByteData;
    FCalcFieldsBuffer: PChar;
    FCheckRowsAffected: Boolean;
    FClonedConnection: TSqlConnection;
    FCommandText: WideString;
    FCommandType: TSQLCommandType;
    FCurrentBlobSize: LongWord;
    FDataLink: TDataLink;
    FDesignerData: string;
    FGetNextRecordSet: Boolean;
    FIndexDefs: TIndexDefs;
    FIndexDefsLoaded: Boolean;
    FLastError: string;  // DBExpress GetError() clears error; need to save last
    FMaxBlobSize: Integer;
    FMaxColSize: LongWord;
    FNativeCommand: WideString;
    FGetMetadata: Boolean;
    FNumericMapping: Boolean;
    FParamCheck: Boolean;
    FParamCount: Integer;
    FParams: TParams;
    FPrepared: Boolean;
    FProcParams: TList;
    FRecords: Integer;
    FRowsAffected: Integer;
    FSchemaInfo: TSQLSchemaInfo;
    FParseSelectSql: TParseSqlEvent;
    FParseUpdateSql: TParseSqlEvent;
    FParseDeleteSql: TParseSqlEvent;
    FParseInsertSql: TParseInsertSqlEvent;
    FSortFieldNames: WideString;

    FSQLCommand: TISQLCommand;
    FSQLConnection: TSQLConnection;
    FSQLCursor: TISQLCursor;

    FStatementOpen: Boolean;
    FTransactionLevel: SmallInt;
    FSchemaName: string;
    function CheckFieldNames(const FieldNames: WideString): Boolean;
    procedure CheckConnection(eFlag: eConnectFlag);
    function CheckDetail(const SQL: WideString): WideString;
    procedure CheckStatement(ForSchema: Boolean = False);
    function GetCalculatedField(Field: TField; var Buffer: Pointer): Boolean;
    function GetDataSetFromSQL(TableName: WideString): TCustomSQLDataSet;
    function GetProcParams: TList;
    function GetInternalConnection: TSQLConnection;
    function GetObjectProcParamCount: Integer; virtual;
    function GetParamCount: Integer; virtual;
    function GetQueryFromType: WideString; virtual;
    function GetRowsAffected: Integer;
    procedure InitBuffers;
    procedure LoadFieldDef(FieldID: Word; var FldDesc: TFLDDesc); overload;
    procedure ReadDesignerData(Reader: TReader);
    procedure RefreshParams;
    procedure SetConnection(const Value: TSQLConnection); virtual;
    procedure SetCurrentBlobSize(Value: LongWord);
    procedure SetDataSource(Value: TDataSource);
    procedure SetParameters(const Value: TParams);
    procedure SetParamsFromProcedure;
    procedure SetParamsFromSQL(DataSet: TDataSet; bFromFields: Boolean);
    procedure SetPrepared(Value: Boolean);
    procedure SetCommandType(const Value: TSQLCommandType); virtual;
    procedure WriteDesignerData(Writer: TWriter);
    procedure SetSchemaName(const Value: string);
    procedure SetSchemaOption;
  protected
    { IProviderSupport2 }
    procedure PSEndTransaction(Commit: Boolean); override;
    procedure PSExecute; override;
    function PSExecuteStatement(const ASQL: WideString; AParams: TParams;
      ResultSet: Pointer = nil): Integer; override;
    procedure PSGetAttributes(List: TList); override;
    function PSGetDefaultOrder: TIndexDef; override;
    function PSGetKeyFieldsW: WideString; override;
    function PSGetIndexDefs(IndexTypes: TIndexOptions): TIndexDefs; override;
    function PSGetParams: TParams; override;
    function PSGetQuoteCharW: WideString; override;
    function PSGetTableNameW: WideString; override;
    function PSGetUpdateException(E: Exception; Prev: EUpdateError): EUpdateError; override;
    function PSInTransaction: Boolean; override;
    function PSIsSQLBased: Boolean; override;
    function PSIsSQLSupported: Boolean; override;
    procedure PSReset; override;
    procedure PSSetCommandText(const ACommandText: WideString); override;
    procedure PSSetParams(AParams: TParams); override;
    procedure PSStartTransaction; override;
    function PSUpdateRecord(UpdateKind: TUpdateKind; Delta: TDataSet): Boolean; override;
    function PSGetCommandText: string; override;
    function PSGetCommandType: TPSCommandType; override;
  protected
    { implementation of abstract TDataSet methods }
    procedure InternalClose; override;
    procedure InternalHandleException; override;
    procedure InternalInitFieldDefs; override;
    procedure InternalOpen; override;
    function IsCursorOpen: Boolean; override;
  protected
    procedure AddFieldDesc(FieldDescs: TFieldDescList; DescNo: Integer;
        var FieldID: Integer; RequiredFields: TBits; FieldDefs: TFieldDefs); 
    procedure AddIndexDefs(SourceDS: TCustomSQLDataSet; IndexName: string = '') ;
    function Check(status: SQLResult; eType: TSQLExceptionType): SQLResult;
    procedure CheckPrepareError;
    procedure ClearIndexDefs;
    procedure CloseCursor; override;
    procedure CloseStatement;
    procedure DefineProperties(Filer: TFiler); override;
    function ExecSQL(ExecDirect: Boolean = False): Integer; virtual;
    procedure ExecuteStatement;
    procedure FreeCursor;
    procedure FreeBuffers;
    procedure FreeStatement;
    function GetCanModify: Boolean; override;
    function GetDataSource: TDataSource; override;
    procedure GetObjectTypeNames(Fields: TFields);
    procedure GetOutputParams(AProcParams: TList = nil);
    function GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult; override;
    function GetSortFieldNames: WideString;
    procedure InitRecord(Buffer: PChar); override;
    procedure InternalRefresh; override;
    procedure Loaded; override;
    function LocateRecord(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptions; SyncCursor: Boolean): Boolean;
    procedure OpenCursor(InfoQuery: Boolean); override;
    procedure OpenSchema; virtual;
    procedure PropertyChanged;
    procedure SetBufListSize(Value: Integer); override;
    procedure SetCommandText(const Value: WideString); virtual;
    procedure SetFieldData(Field: TField; Buffer: Pointer); override;
    procedure SetParamsFromCursor;
    procedure SetSortFieldNames(Value: WideString);
    procedure SQLError(OpStatus: SQLResult; eType: TSQLExceptionType);
    procedure UpdateIndexDefs; override;
    { protected properties }
    property BlobBuffer: TBlobByteData read FBlobBuffer write FBlobBuffer;
    property CurrentBlobSize: LongWord read FCurrentBlobSize write SetCurrentBlobSize;
    property DataLink: TDataLink read FDataLink;
    property InternalConnection: TSqlConnection read GetInternalConnection;
    property LastError: string read FLastError write FLastError;
    property NativeCommand: WideString read FNativeCommand write FNativeCommand;
    property ProcParams: TList read GetProcParams write FProcParams;
    property RowsAffected: Integer read GetRowsAffected;
    procedure SetMaxBlobSize(MaxSize: Integer);
    procedure SetFCommandText(const Value: string);
    property ParamCount: Integer read GetParamCount;
    property SchemaInfo: TSQLSchemaInfo read FSchemaInfo write FSchemaInfo;
  protected  { publish in TSQLDataSet }
    property CommandType: TSQLCommandType read FCommandType write SetCommandType default ctQuery;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property MaxBlobSize: Integer read FMaxBlobSize write SetMaxBlobSize default 0;
    function GetRecordCount: Integer; override;
    property Params: TParams read FParams write SetParameters;
    property ParamCheck: Boolean read FParamCheck write FParamCheck default True;
    property SortFieldNames: WideString read GetSortFieldNames write SetSortFieldNames;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property CommandText: WideString read FCommandText write SetCommandText;
    function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
    function GetBlobFieldData(FieldNo: Integer; var Buffer: TBlobByteData): Integer; override;
    procedure GetDetailLinkFields(MasterFields, DetailFields: TList); override;
    function GetFieldData(FieldNo: Integer; Buffer: Pointer): Boolean; overload; override;
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; overload; override;
    function GetKeyFieldNames(List: TStrings): Integer; overload;
    function GetKeyFieldNames(List: TWideStrings): Integer; overload;
    function GetQuoteChar: WideString; virtual;
    function ParamByName(const Value: string): TParam;
    procedure PrepareStatement; virtual;
    property IndexDefs: TIndexDefs read FIndexDefs write FIndexDefs;
    function IsSequenced: Boolean; override;
    function Locate(const KeyFields: string; const KeyValues: Variant;
      Options: TLocateOptions): Boolean; override;
    function Lookup(const KeyFields: string; const KeyValues: Variant;
      const ResultFields: string): Variant; override;
    procedure SetSchemaInfo(SchemaType: TSchemaType; SchemaObjectName, SchemaPattern: WideString; PackageName: WideString = '' );
    property Prepared: Boolean read FPrepared write SetPrepared default False;
    property DesignerData: string read FDesignerData write FDesignerData;
    property RecordCount: Integer read GetRecordCount;
    property SQLConnection: TSQLConnection read FSQLConnection write SetConnection;
    property TransactionLevel: SmallInt read FTransactionLevel write FTransactionLevel default 0;
  published
    property ParseSelectSql: TParseSqlEvent read FParseSelectSql write FParseSelectSql;
    property ParseDeleteSql: TParseSqlEvent read FParseDeleteSql write FParseDeleteSql;
    property ParseUpdateSql: TParseSqlEvent read FParseUpdateSql write FParseUpdateSql;
    property ParseInsertSql: TParseInsertSqlEvent read FParseInsertSql write FParseInsertSql;
    property SchemaName: string read FSchemaName write SetSchemaName;
    property GetMetadata: Boolean read FGetMetadata write FGetMetadata default True;
    property NumericMapping: Boolean read FNumericMapping write FNumericMapping default False;
    property ObjectView default False;
    property BeforeOpen;
    property AfterOpen;
    property BeforeClose;
    property AfterClose;
    property BeforeScroll;
    property AfterScroll;
    property BeforeRefresh;
    property AfterRefresh;
    property OnCalcFields;
    property Active default False;
  end;

{ TSQLDataSet }

  TSQLDataSet = class(TCustomSQLDataSet)
  public
    constructor Create(AOwner: TComponent); override;
    function ExecSQL(ExecDirect: Boolean = False): Integer; override;
  published
    property CommandText;
    property CommandType;
    property DataSource;
    property MaxBlobSize;
    property ParamCheck;
    property Params;
    property SortFieldNames;
    property SQLConnection;
  end;

{ TSQLQuery }

  TSQLQuery = class(TCustomSQLDataSet)
  private
    FSQL: TWideStrings;
    FText: string;
    procedure QueryChanged(Sender: TObject);
    procedure SetSQL(Value: TWideStrings);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ExecSQL(ExecDirect: Boolean = False): Integer; override;
    procedure PrepareStatement; override;
    property RowsAffected;
    property Text: string read FText;
  published
    property DataSource;
    property GetMetadata default False;
    property MaxBlobSize;
    property ParamCheck;
    property Params;
    property SQL: TWideStrings read FSQL write SetSQL;
    property SQLConnection;
  end;

{ TSQLStoredProc }

  TSQLStoredProc = class(TCustomSQLDataSet)
  private
    FStoredProcName: WideString;
    FPackageName: WideString;
    procedure SetStoredProcName(Value: WideString);
    procedure SetPackageName(Value: WideString);
  public
    constructor Create(AOwner: TComponent); override;
    function ExecProc: Integer; virtual;
    function NextRecordSet: TCustomSQLDataSet;
    procedure PrepareStatement; override;
  published
    property MaxBlobSize;
    property ParamCheck;
    property Params;
    { SetPackageName set StoredProcName to empty string
      Need to set PackageName 1st, and StoredProcName 2nd.
      Don't change following 2 items order }
    property PackageName: WideString read FPackageName write SetPackageName;
    property SQLConnection;
    property StoredProcName: WideString read FStoredProcName write SetStoredProcName;
  end;

{ TSQLTable }

  TSQLTable = class(TCustomSQLDataSet)
  private
    FIsDetail: Boolean;
    FIndexFields: TList;
    FIndexFieldNames: WideString;
    FIndexName: WideString;
    FMasterLink: TMasterDataLink;
    FTableName: WideString;
    FIndexFieldCount: Integer;
    procedure AddParamsToQuery;
    function GetMasterFields: WideString;
    function GetIndexField(Index: Integer): TField;
    function GetIndexFieldCount: Integer;
    function RefreshIndexFields: Integer;
    procedure SetIndexFieldNames(Value: WideString);
    procedure SetIndexName(Value: WideString);
    procedure SetMasterFields(Value: WideString);
    procedure SetTableName(Value: WideString);
    function GetQueryFromType: WideString; override;
    procedure SetDataSource(Value: TDataSource);
  protected
    procedure OpenCursor(InfoQuery: Boolean); override;
    procedure SetIndexField(Index: Integer; Value: TField);
    property MasterLink: TMasterDataLink read FMasterLink;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DeleteRecords;
    procedure GetIndexNames(List: TWideStrings);
    procedure PrepareStatement; override;
    property IndexFields[Index: Integer]: TField read GetIndexField write SetIndexField;
    property IndexFieldCount: Integer read GetIndexFieldCount;
  published
    property Active default False;
    property IndexFieldNames: WideString read FIndexFieldNames write SetIndexFieldNames;
    property IndexName: WideString read FIndexName write SetIndexName;
    property MasterFields: WideString read GetMasterFields write SetMasterFields;
    property MasterSource: TDataSource read GetDataSource write SetDataSource;
    property MaxBlobSize;
    property SQLConnection;
    property TableName: WideString read FTableName write SetTableName;
  end;

{ Utility Routines }

  procedure LoadParamListItems(Params: TParams; ProcParams: TList);
  procedure FreeProcParams(var ProcParams: TList);
  procedure GetConnectionNames(List: TStrings; Driver: string = ''; DesignMode: Boolean = True);
  procedure GetDriverNames(List: TStrings; DesignMode: Boolean = True);
  function GetDriverRegistryFile(DesignMode: Boolean = False): string;
  function GetConnectionRegistryFile(DesignMode: Boolean = False): string;

type
  TGetDriverFunc = function(SVendorLib, SResourceFile: PChar; out Obj): SQLResult; stdcall;

var
{$IFDEF MSWINDOWS}
  DllGetClassObject: function(const CLSID, IID: TGUID; var Obj): HResult; stdcall;
  procedure RegisterDbXpressLib(GetClassProc: Pointer);
{$ENDIF}
threadvar
  GetDriver: TGetDriverFunc;
  DllHandle: THandle;

implementation

{$IFDEF MSWINDOWS}
uses Registry, SqlConst, DBConsts, IniFiles, DBConnAdmin, Math, FMTBcd, WideStrUtils;
{$ENDIF}
{$IFDEF LINUX}
uses SqlConst, DBConsts, IniFiles, Math, DBConnAdmin, FMTBcd;
{$ENDIF}


{ Utility routines }

procedure CheckObject(const Value: IInterface; const eType: TSQLExceptionType);
var
  Message: string;
begin
  if not Assigned(Value) then
  begin
    case eType of
      exceptConnection: Message := SDBXNOCONNECTION;
      exceptCommand: Message := SDBXNOCOMMAND;
      exceptCursor: Message := SDBXNOCURSOR;
      exceptMetadata: Message := SDBXNOMETAOBJECT;
    end;
    DatabaseError(Message);
  end;
end;

function AddQuoteCharToObjectName(DS : TCustomSQLDataSet; Name: Widestring): Widestring;
var
  Status: SQLResult;
begin
  Result := '';
  Status := DS.GetInternalConnection.FISQLConnection.setStringOption(eConnQualifiedName, Name);
  if Status <> 0 then
    DS.SQLError(Status, exceptConnection);
  SetLength(Result, 256);
  Status := DS.GetInternalConnection.FISQLConnection.getStringOption(eConnQuotedObjectName, Result);
  if Status <> 0 then
    DS.SQLError(Status, exceptConnection);
end;


function GetTableScope(Scope: TTableScopes): LongWord;
begin
  Result := 0;
  if tsTable in Scope then
    Result := Result OR eSQLTable;
  if tsView in Scope then
    Result := Result OR eSQLView;
  if tsSysTable in Scope then
    Result := Result OR eSQLSystemTable;
  if tsSynonym in Scope then
    Result := Result OR eSQLSynonym;
end;

{$IFDEF LINUX}
function CopyConfFile(Source, Target: string): Boolean;
var
  List: TStrings;
  IniIn, IniOut: TMemIniFile;
begin
  List := TStringList.Create;
  try
    IniIn := TMemIniFile.Create(Source);
    try
      IniOut := TMemIniFile.Create(Target);
      try
        IniIn.GetStrings(List);
        IniOut.SetStrings(List);
        IniOut.UpdateFile;
        Result := True;
      finally
        IniOut.Free;
      end;
    finally
      IniIn.Free;
    end;
  finally
    List.Free;
  end;
end;
{$ENDIF}

function GetRegistryFile(Setting, Default: string; DesignMode: Boolean): string;
var
{$IFDEF MSWINDOWS}
  Reg: TRegistry;
{$ENDIF}
{$IFDEF LINUX}
  GlobalFile: string;
{$ENDIF}
begin
  {$IFDEF MSWINDOWS}
  Result := '';
  Reg := TRegistry.Create;
  try
    Reg.RootKey := HKEY_CURRENT_USER;
    if Reg.OpenKeyReadOnly(SDBEXPRESSREG_SETTING) then
      Result := Reg.ReadString(Setting);
  finally
    Reg.Free;
  end;
  if Result = '' then
    Result := ExtractFileDir(ParamStr(0)) + '\' + Default;
  {$ENDIF}
  {$IFDEF LINUX}
  Result := getenv('HOME') + SDBEXPRESSREG_USERPATH + Default;    { do not localize }
  if not FileExists(Result) then
  begin
    GlobalFile := SDBEXPRESSREG_GLOBALPATH + Default + SConfExtension;
    if FileExists(GlobalFile) then
    begin
      if DesignMode then
      begin
        if not CopyConfFile(GlobalFile, Result) then
          DatabaseErrorFmt(SConfFileMoveError, [GlobalFile, Result])
      end else
        Result := GlobalFile;
    end else
      DatabaseErrorFmt(SMissingConfFile, [GlobalFile]);
  end;
  {$ENDIF}
end;

function GetDriverRegistryFile(DesignMode: Boolean = False): string;
begin
  Result := GetRegistryFile(SDRIVERREG_SETTING, sDriverConfigFile, DesignMode);
end;

function GetConnectionRegistryFile(DesignMode: Boolean = False): string;
begin
  Result := GetRegistryFile(SCONNECTIONREG_SETTING, sConnectionConfigFile, DesignMode);
end;

function GetBlobSize(DataSet: TCustomSQLDataSet; FieldNo: Integer): LongWord;
var
  IsNull: LongBool;
  Status: SQLResult;
begin
  Result := 0;
  if not DataSet.EOF then
  begin
    if DataSet.MaxBlobSize = 0 then exit;
    Status := DataSet.FSQLCursor.GetBlobSize(Word(FieldNo), Result, IsNull);
    if Status <> DBXERR_NONE then
      DataSet.SQLError(Status, exceptCursor);
    if IsNull then
      Result := 0;
  end;
  DataSet.CurrentBlobSize := Result;
end;

function NextPiece(Start: WideString; InLiteral: Boolean; QuoteChar: WideChar; EndParam: Boolean = False): Integer;
var
  P: PWideChar;
  Ctr: Integer;
  SearchChars: set of char;
begin
  SearchChars := [' ', ')', ',', '=', ':', '>', '<', #13, #10];
  P := (PWideChar(Start))+1;
  Ctr := 1;
  Result := 0;
  while (Result = 0) and (P^ <> #0) do
  begin
    if (P^ = '''') or (P^ = QuoteChar) then
      InLiteral := not InLiteral
    else
    if not InLiteral and inOpSet(P^, SearchChars) then
    begin
      if EndParam then
      begin
        if not inOpSet(P^, ['=', ':', '<', '>']) then
        begin
          Result := Ctr;
          Inc(Result);
        end
      end else
      begin
        if (P^ = ':') then
        begin
          if inOpSet(P[-1], [' ', ')', ',', '=', '(']) then
            Result := Ctr;
        end
        else if (P[1] = ':') then
        begin
          Result := Ctr;
          Inc(Result);
        end;
      end;
    end;
    Inc(P);
    Inc(Ctr);
  end;
end;

// SqlObjects does not support named params: convert to ?
// if not yet converted
function FixParams(SQL: WideString; Count: Integer; QuoteChar: WideString): Widestring;
var
  Param, Start: Widestring;
  Pos, EndPos: Integer;
  InLiteral: Boolean;
  Q: WideChar;
begin
  Q := PWideChar(QuoteChar)[0];
  if inOpSet(Q, [#0, ' ']) then Q := '''';
  InLiteral := False;
  Start := SQL;
  Pos := NextPiece(Start, InLiteral, Q);
  while Pos > 0 do
  begin
    Start := copy(Start, Pos + 1, Length(Start) - Pos);
    EndPos := NextPiece(Start, InLiteral, Q, True);
    if EndPos = 0 then
      Param := copy(Start, 1, Length(Start))
    else
      Param := copy(Start, 1, EndPos-1);
    SQL := WideStringReplace(SQL, Param, ' ? ', []);
    Pos := NextPiece(Start, InLiteral, Q);
  end;
  Result := SQL;
end;

function GetProfileString(Section, Setting, IniFileName: string): string;
var
  IniFile: TMemIniFile;
  List: TStrings;
begin
  List := TStringList.Create;
  try
    IniFile := TMemIniFile.Create(IniFileName);
    IniFile.ReadSectionValues(Section, List);
    try
      Result := List.Values[ Setting ];
    finally
      IniFile.Free;
    end;
  finally
    List.Free;
  end;
end;

procedure GetDriverNames(List: TStrings; DesignMode: Boolean = True);
var
  ConnectionAdmin: IConnectionAdmin;
begin
  ConnectionAdmin := GetConnectionAdmin;
  ConnectionAdmin.GetDriverNames(List);
end;

procedure GetConnectionNames(List: TStrings; Driver: string = ''; DesignMode: Boolean = True);
var
  I: Integer;
  ConnectionAdmin: IConnectionAdmin;
begin
  ConnectionAdmin := GetConnectionAdmin;
  ConnectionAdmin.GetConnectionNames(List, '');
  if Driver <> '' then
  begin
    List.BeginUpdate;
    try
      I := List.Count - 1;
      while I >= 0 do
      begin
        if AnsiCompareText(GetProfileString(List[I], DRIVERNAME_KEY,
              GetConnectionRegistryFile(DesignMode)), Driver) <> 0 then
           List.Delete(I);
        Dec(I);
      end;
    finally
      List.EndUpdate;
    end;
  end;
end;

procedure GetParamData(Param: TParam; Buffer: Pointer; const DrvLocale: TLocale);
begin
  if Buffer <> nil then
  begin
    with Param do
      if DataType in [ftString, ftFixedChar, ftMemo]  then
      begin
        NativeStr := VarToStr(Value);                                              
        GetData(Buffer);
      end
      else
        GetData(Buffer);
  end;
end;


procedure CalcUnits( const Params: TParams; const ProcParams: TList;
          const Index: Integer; pArgDesc: pSPParamDesc; var ChildPos: array of Word );
var
  I: Integer;
  ArgDesc: SPParamDesc;
begin
  I := Index + 1;
  ArgDesc := pArgDesc^;
  pArgDesc.iUnits1 := 0;
  pArgDesc.iUnits2 := 0;
  while (I < Params.Count) do
  begin
    if ProcParams <> nil then
      ArgDesc := (PSPParamDesc(ProcParams.Items[I]))^
    else
      begin
        with ArgDesc, Params[i] do
          begin
            iParamNum := ID + 1;
            szName := Name;
            iArgType := ParamType;
            iDataType := DataType;
            iUnits1 := Precision;
            iUnits2 := NumericScale;
            iLen := GetDataSize;
          end;
      end;
    if ArgDesc.iParamNum <> pArgDesc.iParamNum then
      break;
    Inc(pArgDesc.iUnits1);
    Inc(pArgDesc.iUnits2);
    ChildPos[I] := I - Index;
    if ArgDesc.iDataType = ftADT then
    begin
      CalcUnits(Params, ProcParams, I, @ArgDesc, ChildPos);
      Inc(pArgDesc.iUnits2, ArgDesc.iUnits2);
      Inc(I, ArgDesc.iUnits2);
    end else
      Inc(I);
  end;
end;

procedure SetQueryProcParams(const Sender: TSQLConnection;
  const Command: TISQLCommand; const Params: TParams; ProcParams: TList = nil);
var
  I, IInd, DataLen: Integer;
  iFldNum: LongWord;
  RecBuffer: PWideChar;
  iFldType, iSubType: Word;
  DrvLocale: TLocale;
  Status: SQLResult;
  ArgDesc: SPParamDesc;
  ChildPosArray: array of Word;
  SBcd: string;
  Bcd: TBcd;
begin
  DrvLocale := nil;
  SetLength(ChildPosArray, Params.Count);
  for I := 0 to Params.Count - 1 do
    begin
      RecBuffer := nil;
      try
        if Params[I].ParamType = ptUnknown then  // Midas assumes its Input
          Params[I].ParamType := ptInput;
        iFldNum := i + 1;
        iFldType := FldTypeMap[Params[I].DataType];
        iSubType := 0;
        if iFldType in [fldBlob, fldZString] then
          iSubType := Word(FldSubTypeMap[Params[I].DataType])
        else if iFldType = fldUNKNOWN then
          DatabaseErrorFmt(SNoParameterValue, [Params[I].Name]);
        if ProcParams <> nil then
          ArgDesc := (PSPParamDesc(ProcParams.Items[I]))^
        else
          with ArgDesc, Params[i] do
            begin
              iParamNum := iFldNum;
              szName := Name;
              iArgType := ParamType;
              iDataType := DataType;
              iUnits1 := Precision;
              iUnits2 := NumericScale;
              iLen := GetDataSize;
            end;
        iFldType := FldTypeMap[ArgDesc.iDataType];
        if Params[I].ParamType <> ptOutput then
          DataLen := Params[I].GetDataSize
        else
          DataLen := ArgDesc.iLen;
        {Check if the IN param is NULL and set the NULL indicator}
        if ((Params[I].ParamType = ptInput) and Params[I].IsNull) then
          iInd := 1
        else
        if (DataLen > 0) then
        begin
          iInd := 0;
          RecBuffer := AllocMem(DataLen);
          if Params[I].ParamType <> ptOutput then
            GetParamData(Params[I], RecBuffer, DrvLocale)
          else
            FillChar(RecBuffer^, DataLen, 0);
          if Params[I].ParamType = ptInput then
            Params[I].Size := 0;
          if (Params[I].ParamType = ptOutput) and not(iFldType in [fldFLOAT]) then
            ArgDesc.iLen := 0
          else
            case iFldType of
              fldBlob:
                 begin
                  ArgDesc.iLen := DataLen;
                  ArgDesc.iUnits2 := 0;
                  if ( (iSubType = fldstMemo) or (iSubType = fldstHMemo) or 
                       (iSubType = fldstWideMemo) ) then
                    begin
                      if (DataLen > 0 ) then
                      begin
                        Params[I].Size := DataLen - 1; //Max precision
                        ArgDesc.iLen := DataLen -1;    //Length
                      end;
                    end;
                 end;
              fldZString, fldBYTES, fldVARBYTES:
                begin
                  ArgDesc.iLen := DataLen;
                  ArgDesc.iUnits2 := 0;
                  
                  //Handle ptInput     
                  if (Params[I].ParamType = ptInput) then
                  begin
                    if iFldType = fldVARBYTES then
                      Params[I].Size := DataLen - 2
                    else if iFldType = fldZString then
                    begin
                      if (DataLen > 0 ) then
                        Params[I].Size := DataLen - 1
                    end
                    else
                      Params[I].Size := DataLen;
                  end;
                  //Handle ptInput     

                  if (Params[I].ParamType = ptInputOutput) and (DataLen > Params[I].Size) then
                  begin
                    if iFldType = fldVARBYTES then
                      Params[I].Size := DataLen - 2
                    else if iFldType = fldZString then
                      Params[I].Size := DataLen - 1
                    else
                      Params[I].Size := DataLen;
                  end;
                end;
              fldFLOAT:
                begin
                  if Params[I].Precision = 4 then
                    ArgDesc.iLen := 4
                  else
                    ArgDesc.iLen := Sizeof(Double);
                end;
              fldFMTBCD, fldBCD:
                begin
                  iFldType := fldBCD;   { DBExpress does not distinguish }
                  if Params[I].Size = 0 then
                  begin
                    SBcd := BcdToStr(PBcd(RecBuffer)^);
                    Bcd := StrToBcd(SBcd);
                    Params[I].Size := Bcd.Precision;
                    ArgDesc.iUnits2 := Bcd.SignSpecialPlaces AND $3F;
                  end else
                  begin
                    ArgDesc.iUnits2 := Params[I].NumericScale;
                  end;
                end;
              fldADT, fldARRAY:
                begin
                  CalcUnits(Params, ProcParams, I, @ArgDesc, ChildPosArray);
                  ArgDesc.iLen := DataLen;
                end;
            end;
        end else  // leave RecBuffer nil
        begin
          if iFldType in [fldADT, fldARRAY] then
            DatabaseError(SObjectTypenameRequired);
          iInd := 1;
        end;
        Status := Command.setParameter(iFldNum - ChildPosArray[I], ChildPosArray[I], TSTMTParamType(ArgDesc.iArgType),
                iFldType, iSubType, Params[I].Size,
                Integer(ArgDesc.iUnits2), ArgDesc.iLen, RecBuffer, IInd);
        if (Status <> DBXERR_NONE) then
          Sender.SQLError(Status, exceptConnection);
      finally
        if RecBuffer <> nil then FreeMem(RecBuffer);
      end;
    end;
end;

procedure FreeProcParams(var ProcParams: TList);
var
  ArgParam: pSPParamDesc;
  I: Integer;
begin
  if not Assigned(ProcParams) then Exit;
  for I := 0 to ProcParams.Count -1 do
  begin
    ArgParam := ProcParams[I];
    Dispose(ArgParam);
  end;
  FreeAndNil(ProcParams);
end;

procedure LoadParamListItems(Params: TParams; ProcParams: TList);
var
  I: Integer;
  ArgParam: pSPParamDesc;
begin
  for I := 0 to ProcParams.Count -1 do
  begin
    ArgParam := ProcParams.Items[I];
    with TParam(Params.Add) do
    begin
      Name := ArgParam.szName;
      ParamType := ArgParam.iArgType;
      DataType := ArgParam.iDataType;
      if ParamType <> ptInput then
        Size := ArgParam.iLen;
    end;
  end;
end;

{ TSQLBlobStream }

constructor TSQLBlobStream.Create(Field: TBlobField; Mode: TBlobStreamMode = bmRead);
begin
  if not Field.DataSet.Active then
    DataBaseError(SDatasetClosed);
  FField := Field;
  FDataSet := FField.DataSet as TCustomSQLDataSet;
  FFieldNo := FField.FieldNo;
  ReadBlobData;
end;

destructor TSQLBlobStream.Destroy;
begin
  inherited Destroy;
end;

procedure TSQLBlobStream.ReadBlobData;
var
  BlobLength: LongInt;
begin
  Clear;
  BlobLength := GetBlobSize(FDataSet, FFieldNo);
  SetSize(BlobLength);
  if BlobLength = 0 then Exit;
  if FDataSet.GetFieldData(FField, FDataSet.FBlobBuffer, True) then
    Write(pointer(FDataSet.FBlobBuffer)^, FDataSet.FCurrentBlobSize);
  Position := 0;
end;

{ Forward declear }

type
  TISQLConnection30 = class;
  TISQLConnection25 = class;
  TISQLCommand30 = class;
  TISQLCommand25 = class;
  TISQLCursor30 = class;
  TISQLCursor25 = class;
  TISQLMetaData30 = class;
  TISQLMetaData25 = class;
  TFLDDesc30 = class;
  TFLDDesc25 = class;

{ TISQLConnection30 }

 TISQLConnection30 = class(TISQLConnection)
  private
    I: ISQLConnection30;
  public
    constructor Create(NewConnection: ISQLConnection); override;
    destructor Destroy; override;
    function connect(): SQLResult; overload; override;
    function connect(ServerName: PWideChar; UserName: PWideChar;
                          Password: PWideChar): SQLResult; overload; override;
    function disconnect: SQLResult; override;
    function getSQLCommand(var pComm: TISQLCommand): SQLResult; override;
    function getSQLMetaData(var pMetaData: TISQLMetaData): SQLResult; override;
    function SetOption(eConnectOption: TSQLConnectionOption;
            lValue: LongInt): SQLResult; override;
    function SetStringOption(eConnectOption: TSQLConnectionOption;
            const lValue: WideString): SQLResult; override;
    function GetOption(eDOption: TSQLConnectionOption; PropValue: Pointer;
            MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eDOption: TSQLConnectionOption;
            var str: WideString): SQLResult; override;
    function beginTransaction(TranID: LongWord): SQLResult; override;
    function commit(TranID: LongWord): SQLResult; override;
    function rollback(TranID: LongWord): SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
    function getFldDescClass: TFldDescRef; override;
  end;

{ TISQLConnection25 }

  TISQLConnection25 = class(TISQLConnection)
  private
    I: ISQLConnection25;
  public
    constructor Create(NewConnection: ISQLConnection); override;
    destructor Destroy; override;
    function connect(): SQLResult; overload; override;
    function connect(ServerName: PWideChar; UserName: PWideChar;
                          Password: PWideChar): SQLResult; overload; override;
    function disconnect: SQLResult; override;
    function getSQLCommand(var pComm: TISQLCommand): SQLResult; override;
    function getSQLMetaData(var pMetaData: TISQLMetaData): SQLResult; override;
    function SetOption(eConnectOption: TSQLConnectionOption;
            lValue: LongInt): SQLResult; override;
    function SetStringOption(eConnectOption: TSQLConnectionOption;
            const lValue: WideString): SQLResult; override;
    function GetOption(eDOption: TSQLConnectionOption; PropValue: Pointer;
            MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eDOption: TSQLConnectionOption;
            var str: WideString): SQLResult; override;
    function beginTransaction(TranID: LongWord): SQLResult; override;
    function commit(TranID: LongWord): SQLResult; override;
    function rollback(TranID: LongWord): SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
    function getFldDescClass: TFldDescRef; override;
  end;

{ TISQLCommand30 }

  TISQLCommand30 = class(TISQLCommand)
  private
    I : ISQLCommand30;
  public
    constructor Create(newCommand: ISQLCommand30);
    destructor Destroy; override;
    function SetOption(
      eSqlCommandOption: TSQLCommandOption;
      ulValue: Integer): SQLResult; override;
    function SetStringOption(
      eSqlCommandOption: TSQLCommandOption;
      const ulValue: WideString): SQLResult; override;
    function GetOption(eSqlCommandOption: TSQLCommandOption;
      PropValue: Pointer;
      MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eSqlCommandOption: TSQLCommandOption;
      var str: WideString): SQLResult; override;
    function setParameter(
      ulParameter: Word ;
      ulChildPos: Word ;
      eParamType: TSTMTParamType ;
      uLogType: Word;
      uSubType: Word;
      iPrecision: Integer;
      iScale: Integer;
      Length: LongWord ;
      pBuffer: Pointer;
      lInd: Integer): SQLResult; override;
    function getParameter(ParameterNumber: Word; ulChildPos: Word; Value: Pointer;
      Length: Integer; var IsBlank: Integer): SQLResult; override;
    function prepare(SQL: PWideChar; ParamCount: Word): SQLResult; override;
    function execute(var Cursor: TISQLCursor): SQLResult; override;
    function executeImmediate(SQL: PWideChar; var Cursor: TISQLCursor): SQLResult; override;
    function getNextCursor(var Cursor: TISQLCursor): SQLResult; override;
    function getRowsAffected(var Rows: LongWord): SQLResult; override;
    function close: SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
  end;

{ TISQLCommand25 }

  TISQLCommand25 = class(TISQLCommand)
  private
    I : ISQLCommand25;
  public
    constructor Create(NewCommand: ISQLCommand25);
    destructor Destroy; override;
    function SetOption(
      eSqlCommandOption: TSQLCommandOption;
      ulValue: Integer): SQLResult; override;
    function SetStringOption(
      eSqlCommandOption: TSQLCommandOption;
      const ulValue: WideString): SQLResult; override;
    function GetOption(eSqlCommandOption: TSQLCommandOption;
      PropValue: Pointer;
      MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eSqlCommandOption: TSQLCommandOption;
      var str: WideString): SQLResult; override;
    function setParameter(
      ulParameter: Word ;
      ulChildPos: Word ;
      eParamType: TSTMTParamType ;
      uLogType: Word;
      uSubType: Word;
      iPrecision: Integer;
      iScale: Integer;
      Length: LongWord ;
      pBuffer: Pointer;
      lInd: Integer): SQLResult; override;
    function getParameter(ParameterNumber: Word; ulChildPos: Word; Value: Pointer;
      Length: Integer; var IsBlank: Integer): SQLResult; override;
    function prepare(SQL: PWideChar; ParamCount: Word): SQLResult; override;
    function execute(var Cursor: TISQLCursor): SQLResult; override;
    function executeImmediate(SQL: PWideChar; var Cursor: TISQLCursor): SQLResult; override;
    function getNextCursor(var Cursor: TISQLCursor): SQLResult; override;
    function getRowsAffected(var Rows: LongWord): SQLResult; override;
    function close: SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
  end;

{ TTISQLCursor30 }

  TISQLCursor30 = class(TISQLCursor)
  private
    I : ISQLCursor30;
  public
    constructor Create(NewCursor: ISQLCursor30);
    destructor Destroy; override;
    function SetOption(eOption: TSQLCursorOption;
                     PropValue: LongInt): SQLResult; override;
    function SetStringOption(eOption: TSQLCursorOption;
                     const str: WideString): SQLResult; override;
    function GetOption(eOption: TSQLCursorOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eOption: TSQLCursorOption;
                     var str: WideString): SQLResult; override;
    function getCurObjectTypeName(const iFldNum: Word): WideString; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload;  override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;  override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
    function getColumnCount(var pColumns: Word): SQLResult;  override;
    function getColumnNameLength(
      ColumnNumber: Word;
      var pLen: Word): SQLResult;  override;
    function getColumnName(ColumnNumber: Word; pColumnName: PWideChar): SQLResult; overload; override;
    function getColumnName(ColumnNumber: Word): WideString; overload; override;
    function getColumnType(ColumnNumber: Word; var puType: Word;
      var puSubType: Word): SQLResult;  override;
    function  getColumnLength(ColumnNumber: Word; var pLength: LongWord): SQLResult;  override;
    function getColumnPrecision(ColumnNumber: Word;
      var piPrecision: SmallInt): SQLResult;  override;
    function getColumnScale(ColumnNumber: Word; var piScale: SmallInt): SQLResult;  override;
    function isNullable(ColumnNumber: Word; var Nullable: LongBool): SQLResult;  override;
    function isAutoIncrement(ColumnNumber: Word; var AutoIncr: LongBool): SQLResult;  override;
    function isReadOnly(ColumnNumber: Word; var ReadOnly: LongBool): SQLResult; override;
    function isSearchable(ColumnNumber: Word; var Searchable: LongBool): SQLResult; override;
    function isBlobSizeExact(ColumnNumber: Word; var IsExact: LongBool): SQLResult; override;
    function next: SQLResult; override;
    function getString(ColumnNumber: Word; Value: PChar;
      var IsBlank: LongBool): SQLResult; override;
    function getWideString(ColumnNumber: Word; Value: PWideChar;
      var IsBlank: LongBool): SQLResult; override;
    function getShort(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getLong(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getInt64(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getDouble(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBcd(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getTimeStamp(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getTime(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getDate(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBytes(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBlobSize(ColumnNumber: Word; var Length: LongWord;
      var IsBlank: LongBool): SQLResult; override;
    function getBlob(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool; Length: LongWord): SQLResult; override;
    property cursor: ISQLCursor30 read I;
end;

{ TTISQLCursor25 }

  TISQLCursor25 = class(TISQLCursor)
  private
    I : ISQLCursor25;
  public
    constructor Create(NewCursor: ISQLCursor25);
    destructor Destroy; override;
    function SetOption(eOption: TSQLCursorOption;
                     PropValue: LongInt): SQLResult; override;
    function SetStringOption(eOption: TSQLCursorOption;
                     const str: WideString): SQLResult; override;
    function GetOption(eOption: TSQLCursorOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eOption: TSQLCursorOption;
                     var str: WideString): SQLResult; override;
    function getCurObjectTypeName(const iFldNum: Word): WideString; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload;  override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;  override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
    function getColumnCount(var pColumns: Word): SQLResult;  override;
    function getColumnNameLength(
      ColumnNumber: Word;
      var pLen: Word): SQLResult;  override;
    function getColumnName(ColumnNumber: Word; pColumnName: PWideChar): SQLResult; overload; override;
    function getColumnName(ColumnNumber: Word): WideString; overload; override;
    function getColumnType(ColumnNumber: Word; var puType: Word;
      var puSubType: Word): SQLResult;  override;
    function  getColumnLength(ColumnNumber: Word; var pLength: LongWord): SQLResult;  override;
    function getColumnPrecision(ColumnNumber: Word;
      var piPrecision: SmallInt): SQLResult;  override;
    function getColumnScale(ColumnNumber: Word; var piScale: SmallInt): SQLResult;  override;
    function isNullable(ColumnNumber: Word; var Nullable: LongBool): SQLResult;  override;
    function isAutoIncrement(ColumnNumber: Word; var AutoIncr: LongBool): SQLResult;  override;
    function isReadOnly(ColumnNumber: Word; var ReadOnly: LongBool): SQLResult; override;
    function isSearchable(ColumnNumber: Word; var Searchable: LongBool): SQLResult; override;
    function isBlobSizeExact(ColumnNumber: Word; var IsExact: LongBool): SQLResult; override;
    function next: SQLResult; override;
    function getString(ColumnNumber: Word; Value: PChar;
      var IsBlank: LongBool): SQLResult; override;
    function getWideString(ColumnNumber: Word; Value: PWideChar;
      var IsBlank: LongBool): SQLResult; override;
    function getShort(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getLong(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getInt64(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getDouble(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBcd(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getTimeStamp(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getTime(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getDate(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBytes(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult; override;
    function getBlobSize(ColumnNumber: Word; var Length: LongWord;
      var IsBlank: LongBool): SQLResult; override;
    function getBlob(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool; Length: LongWord): SQLResult; override;
end;

{ TISQLMetaData30 }

  TISQLMetaData30 = class(TISQLMetaData)
  protected
    I : ISQLMetaData30;
  public
    constructor Create(newMetaData: ISQLMetaData30);
    destructor Destroy; override;
    function SetOption(eDOption: TSQLMetaDataOption;
                     PropValue: LongInt): SQLResult; override;
    function SetStringOption(eDOption: TSQLMetaDataOption;
                     const str: WideString): SQLResult; override;
    function GetOption(eDOption: TSQLMetaDataOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eDOption: TSQLMetaDataOption;
                     var str: WideString): SQLResult; override;
    function getObjectList(eObjType: TSQLObjectType; var Cursor: TISQLCursor):
                     SQLResult; override;
    function getTables(TableName: PWideChar; TableType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getProcedures(ProcedureName: PWideChar; ProcType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getColumns(TableName: PWideChar; ColumnName: PWideChar;
                     ColType: LongWord; var Cursor: TISQLCursor): SQLResult; override;
    function getProcedureParams(ProcName: PWideChar; ParamName: PWideChar;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getIndices(TableName: PWideChar; IndexType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
  end;

{ TISQLMetaData25 }

  TISQLMetaData25 = class(TISQLMetaData)
  protected
    I : ISQLMetaData25;
  public
    constructor Create(newMetaData: ISQLMetaData25);
    destructor Destroy; override;
    function SetOption(eDOption: TSQLMetaDataOption;
                     PropValue: LongInt): SQLResult; override;
    function SetStringOption(eDOption: TSQLMetaDataOption;
                     const str: WideString): SQLResult; override;
    function GetOption(eDOption: TSQLMetaDataOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult; override;
    function GetStringOption(eDOption: TSQLMetaDataOption;
                     var str: WideString): SQLResult; override;
    function getObjectList(eObjType: TSQLObjectType; var Cursor: TISQLCursor):
                     SQLResult; override;
    function getTables(TableName: PWideChar; TableType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getProcedures(ProcedureName: PWideChar; ProcType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getColumns(TableName: PWideChar; ColumnName: PWideChar;
                     ColType: LongWord; var Cursor: TISQLCursor): SQLResult; override;
    function getProcedureParams(ProcName: PWideChar; ParamName: PWideChar;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getIndices(TableName: PWideChar; IndexType: LongWord;
                     var Cursor: TISQLCursor): SQLResult; override;
    function getErrorMessage(Error: PWideChar): SQLResult; overload; override;
    function getErrorMessageLen(out ErrorLen: SmallInt): SQLResult; override;
    function getErrorMessage(var Error: WideString): SQLResult; overload; override;
  end;

{ TFLDDesc30 }

  TFLDDesc30 = class(TFLDDesc)
  private
    FFLDDesc30: FLDDesc30;
  protected
    function GetiFldNum: Word; override;
    function GetszName: WideString; override;
    function GetiFldType: Word; override;
    function GetiSubType: Word; override;
    function GetiUnits1: SmallInt; override;
    function GetiUnits2: SmallInt; override;
    function GetiOffset: Word; override;
    function GetiLen: LongWord; override;
    function GetiNullOffset: Word; override;
    function GetefldvVchk: FLDVchk; override;
    function GetefldrRights: FLDRights; override;
    function GetbCalcField: WordBool; override;
    procedure SetiFldNum(Value: Word); override;
    procedure SetszName(Value: WideString); override;
    procedure SetiFldType(Value: Word); override;
    procedure SetiSubType(Value: Word); override;
    procedure SetiUnits1(Value: SmallInt); override;
    procedure SetiUnits2(Value: SmallInt); override;
    procedure SetiOffset(Value: Word); override;
    procedure SetiLen(Value: LongWord); override;
    procedure SetiNullOffset(Value: Word); override;
    procedure SetefldvVchk(Value: FLDVchk); override;
    procedure SetefldrRights(Value: FLDRights); override;
    procedure SetbCalcField(Value: WordBool); override;
  public
    property desc: FLDDesc30 read FFLDDesc30 write FFLDDesc30;
  end;

{ TFLDDesc25 }

  TFLDDesc25 = class(TFLDDesc)
  private
    FFLDDesc25: FLDDesc25;
  protected
    function GetiFldNum: Word; override;
    function GetszName: WideString; override;
    function GetiFldType: Word; override;
    function GetiSubType: Word; override;
    function GetiUnits1: SmallInt; override;
    function GetiUnits2: SmallInt; override;
    function GetiOffset: Word; override;
    function GetiLen: LongWord; override;
    function GetiNullOffset: Word; override;
    function GetefldvVchk: FLDVchk; override;
    function GetefldrRights: FLDRights; override;
    function GetbCalcField: WordBool; override;
    procedure SetiFldNum(Value: Word); override;
    procedure SetszName(Value: WideString); override;
    procedure SetiFldType(Value: Word); override;
    procedure SetiSubType(Value: Word); override;
    procedure SetiUnits1(Value: SmallInt); override;
    procedure SetiUnits2(Value: SmallInt); override;
    procedure SetiOffset(Value: Word); override;
    procedure SetiLen(Value: LongWord); override;
    procedure SetiNullOffset(Value: Word); override;
    procedure SetefldvVchk(Value: FLDVchk); override;
    procedure SetefldrRights(Value: FLDRights); override;
    procedure SetbCalcField(Value: WordBool); override;
  public
    property desc: FLDDesc25 read FFLDDesc25 write FFLDDesc25;
  end;

{ TISQLConnection30 }

constructor TISQLConnection30.Create(NewConnection: ISQLConnection);
begin
  I := ISQLConnection30(NewConnection);
end;

destructor TISQLConnection30.Destroy;
begin
  I := Nil;
  Inherited;
end;

function TISQLConnection30.connect(): SQLResult;
begin
  result := I.connect();
end;

function TISQLConnection30.connect(ServerName: PWideChar; UserName: PWideChar; Password: PWideChar): SQLResult;
begin
  result := I.connect(ServerName, UserName, Password);
end;

function TISQLConnection30.disconnect: SQLResult;
begin
  result := I.disconnect;
end;

function TISQLConnection30.getSQLCommand(var pComm: TISQLCommand): SQLResult;
var
  pComm30: ISQLCommand30;
begin
  result := I.getSQLCommand(pComm30);
  if (pComm30 <> Nil) then
    pComm := TISQLCommand30.create(pComm30)
  else
    pComm := Nil;
end;

function TISQLConnection30.getSQLMetaData(var pMetaData: TISQLMetaData): SQLResult;
var
  pMetaData30: ISQLMetaData30;
begin
  result := I.getSQLMetaData(pMetaData30);
  if (pMetaData30 <> nil) then
    pMetaData := TISQLMetaData30.Create(pMetaData30)
  else
    pMetaData := Nil;
end;

function TISQLConnection30.SetOption(eConnectOption: TSQLConnectionOption; lValue: LongInt): SQLResult;
begin
  result := I.SetOption(eConnectOption, lValue);
end;

function TISQLConnection30.SetStringOption(eConnectOption: TSQLConnectionOption;
            const lValue: WideString): SQLResult;
begin
  result := I.SetOption(eConnectOption, Integer(PWideChar(lValue)));
end;

function TISQLConnection30.GetOption(eDOption: TSQLConnectionOption; PropValue: Pointer;
            MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  result := I.GetOption(eDOption, PropValue, MaxLength, Length);
end;

function TISQLConnection30.GetStringOption(eDOption: TSQLConnectionOption;
            var str: WideString): SQLResult;
var
  Len: SmallInt;
begin
  Len := Length(str);
  FillChar(str[1], Len, #0);
  result := I.GetOption(eDOption, PWideChar(str), Len * sizeof(widechar), Len);
  SetLength(str, WStrLen(PWideChar(str)));
end;

function TISQLConnection30.beginTransaction(TranID: LongWord): SQLResult;
begin
  result := I.beginTransaction(TranID);
end;

function TISQLConnection30.commit(TranID: LongWord): SQLResult;
begin
  result := I.commit(TranID);
end;

function TISQLConnection30.rollback(TranID: LongWord): SQLResult;
begin
  result := I.rollback(TranID);
end;

function TISQLConnection30.getErrorMessage(Error: PWideChar): SQLResult;
begin
  result := I.getErrorMessage(Error);
end;

function TISQLConnection30.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLConnection30.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PWideChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1) * SizeOf(WideChar));
    result := I.getErrorMessage(Message);
  end;
  if (result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

function TISQLConnection30.getFldDescClass: TFldDescRef;
begin
  Result := TFLDDesc30;
end;

{ TISQLConnection25 }

constructor TISQLConnection25.Create(NewConnection: ISQLConnection);
begin
  I := ISQLConnection25(NewConnection);
end;

destructor TISQLConnection25.Destroy;
begin
  I := Nil;
  Inherited;
end;

function TISQLConnection25.connect(): SQLResult;
begin
  Result := DBXERR_SQLERROR;
end;

function TISQLConnection25.connect(ServerName: PWideChar; UserName: PWideChar; Password: PWideChar): SQLResult;
var
  aServerName, aUserName, aPassword : AnsiString;
begin
  aServerName := ServerName;
  aUserName := UserName;
  aPassword := Password;
  result := I.connect(PChar(aServerName), PChar(aUserName), PChar(aPassword));
end;

function TISQLConnection25.disconnect: SQLResult;
begin
  result := I.disconnect;
end;

function TISQLConnection25.getSQLCommand(var pComm: TISQLCommand): SQLResult;
var
  pComm25: ISQLCommand25;
begin
  result := I.getSQLCommand(pComm25);
  if (pComm25 <> Nil) then
    pComm := TISQLCommand25.create(pComm25)
  else
    pComm := Nil;
end;

function TISQLConnection25.getSQLMetaData(var pMetaData: TISQLMetaData): SQLResult;
var
  pMetaData25: ISQLMetaData25;
begin
  result := I.getSQLMetaData(pMetaData25);
  if (pMetaData25 <> Nil) then
    pMetaData := TISQLMetaData25.Create(pMetaData25)
  else
    pMetaData := Nil;
end;

function TISQLConnection25.SetOption(eConnectOption: TSQLConnectionOption; lValue: LongInt): SQLResult;
begin
  result := I.SetOption(eConnectOption, lValue);
end;

function TISQLConnection25.SetStringOption(eConnectOption: TSQLConnectionOption;
            const lValue: WideString): SQLResult;
begin
  result := I.SetOption(eConnectOption, integer(PChar(ANsiString(lValue))));
end;

function TISQLConnection25.GetOption(eDOption: TSQLConnectionOption; PropValue: Pointer;
            MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  result := I.GetOption(eDOption, PropValue, MaxLength, Length);
end;

function TISQLConnection25.GetStringOption(eDOption: TSQLConnectionOption;
            var str: WideString): SQLResult;
var
  aStr: AnsiString;
  Len, OrgLen: SmallInt;
begin
  OrgLen := Length(str);
  SetLength(aStr, OrgLen * sizeof(widechar));
  FillChar(aStr[1], OrgLen * sizeof(widechar), #0);
  result := I.GetOption(eDOption, PAnsiChar(aStr), OrgLen * sizeof(widechar), Len);
  SetLength(aStr, strlen(PChar(aStr)));
  str := Copy(aStr, 1, OrgLen);
end;

function TISQLConnection25.beginTransaction(TranID: LongWord): SQLResult;
begin
  result := I.beginTransaction(TranID);
end;

function TISQLConnection25.commit(TranID: LongWord): SQLResult;
begin
  result := I.commit(TranID);
end;

function TISQLConnection25.rollback(TranID: LongWord): SQLResult;
begin
  result := I.rollback(TranID);
end;

function TISQLConnection25.getErrorMessage(Error: PWideChar): SQLResult;
begin
  result := I.getErrorMessage(PChar(Error));
end;

function TISQLConnection25.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLConnection25.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (Result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1));
    Result := I.getErrorMessage(Message);
  end;
  if (Result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

function TISQLConnection25.getFldDescClass: TFldDescRef;
begin
  Result := TFLDDesc25;
end;

{ TISQLCommand }

destructor TISQLCommand.Destroy;
begin
  if Assigned(FSQLCursor) then
    FreeAndNil(FSQLCursor);
  inherited;
end;

{ TISQLCommand30 }

constructor TISQLCommand30.Create(newCommand: ISQLCommand30);
begin
  I := newCommand;
end;

destructor TISQLCommand30.Destroy;
begin
  I := Nil;
  inherited;
end;

function TISQLCommand30.SetOption(
      eSqlCommandOption: TSQLCommandOption;
      ulValue: Integer): SQLResult;
begin
  Result := I.SetOption(eSqlCommandOption, ulValue);
end;

function TISQLCommand30.SetStringOption(
      eSqlCommandOption: TSQLCommandOption;
      const ulValue: WideString): SQLResult;
begin
  Result := I.SetOption(eSqlCommandOption, Integer(PWideChar(ulValue)));
end;

function TISQLCommand30.GetOption(eSqlCommandOption: TSQLCommandOption;
      PropValue: Pointer;
      MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  Result := I.GetOption(eSqlCommandOption, PropValue, MaxLength, Length);
end;

function TISQLCommand30.GetStringOption(eSqlCommandOption: TSQLCommandOption;
      var str: WideString): SQLResult;
var
  Len: SmallInt;
begin
  Len := Length(str) * sizeof(WideChar);
  FillChar(str[1], Len, #0);
  result := I.GetOption(eSqlCommandOption , PWideChar(str), Len * sizeof(widechar), Len);
  SetLength(str, WStrLen(PWideChar(str)));
end;

function TISQLCommand30.setParameter(
      ulParameter: Word ;
      ulChildPos: Word ;
      eParamType: TSTMTParamType ;
      uLogType: Word;
      uSubType: Word;
      iPrecision: Integer;
      iScale: Integer;
      Length: LongWord ;
      pBuffer: Pointer;
      lInd: Integer): SQLResult;
begin
  Result := I.setParameter(ulParameter, ulChildPos, eParamType, uLogType,
      uSubType, iPrecision, iScale, Length, pBuffer, lInd);
end;

function TISQLCommand30.getParameter(ParameterNumber: Word; ulChildPos: Word; Value: Pointer;
      Length: Integer; var IsBlank: Integer): SQLResult;
begin
  Result := I.getParameter(ParameterNumber, ulChildPos, Value, Length, IsBlank);
end;

function TISQLCommand30.prepare(SQL: PWideChar; ParamCount: Word): SQLResult;
begin
  Result := I.prepare(SQL, ParamCount);
end;

function TISQLCommand30.execute(var Cursor: TISQLCursor): SQLResult;
var
  newCursor: ISQLCursor30;
begin
  result := I.execute(newCursor);
  if (newCursor <> nil) then
    Cursor := TISQLCursor30.Create(newCursor)
  else
    Cursor := nil;
end;

function TISQLCommand30.executeImmediate(SQL: PWideChar; var Cursor: TISQLCursor): SQLResult;
var
  newCursor: ISQLCursor30;
begin
  Result := I.executeImmediate(SQL, newCursor);
  if (newCursor <> Nil) then
    Cursor := TISQLCursor30.Create(newCursor)
  else
    Cursor := Nil;
end;

function TISQLCommand30.getNextCursor(var Cursor: TISQLCursor): SQLResult;
var
  newCursor: ISQLCursor30;
begin
  Result := I.getNextCursor(newCursor);
  if (newCursor <> Nil) then
    Cursor := TISQLCursor30.Create(newCursor)
  else
    Cursor := Nil;
end;

function TISQLCommand30.getRowsAffected(var Rows: LongWord): SQLResult;
begin
  Result := I.getRowsAffected(Rows);
end;

function TISQLCommand30.close: SQLResult;
begin
  Result := I.close;
end;

function TISQLCommand30.getErrorMessage(Error: PWideChar): SQLResult;
begin
  Result := I.getErrorMessage(Error);
end;

function TISQLCommand30.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  Result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLCommand30.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PWideChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1) * SizeOf(WideChar));
    result := I.getErrorMessage(Message);
  end;
  if (result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

{ TISQLCommand25 }

constructor TISQLCommand25.Create(NewCommand: ISQLCommand25);
begin
  I := NewCommand;
end;

destructor TISQLCommand25.Destroy;
begin
  I := Nil;
  inherited;
end;

function TISQLCommand25.SetOption(
      eSqlCommandOption: TSQLCommandOption;
      ulValue: Integer): SQLResult;
begin
  Result := I.SetOption(eSqlCommandOption, ulValue);
end;

function TISQLCommand25.SetStringOption(
      eSqlCommandOption: TSQLCommandOption;
      const ulValue: WideString): SQLResult;
begin
  Result := I.SetOption(eSqlCommandOption, Integer(PChar(AnsiString(ulValue))));
end;

function TISQLCommand25.GetOption(eSqlCommandOption: TSQLCommandOption;
      PropValue: Pointer;
      MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  Result := I.GetOption(eSqlCommandOption, PropValue, MaxLength, Length);
end;

function TISQLCommand25.GetStringOption(eSqlCommandOption: TSQLCommandOption;
      var str: WideString): SQLResult;
var
  aStr: AnsiString;
  Len, OrgLen: SmallInt;
begin
  OrgLen := Length(str);
  SetLength(aStr, OrgLen * sizeof(widechar));
  FillChar(aStr[1], OrgLen * sizeof(widechar), #0);
  result := I.GetOption(eSqlCommandOption , PAnsiChar(aStr), OrgLen * sizeof(widechar), Len);
  SetLength(aStr, strlen(PChar(aStr)));
  str := Copy(aStr, 1, OrgLen);
end;

function TISQLCommand25.setParameter(
      ulParameter: Word ;
      ulChildPos: Word ;
      eParamType: TSTMTParamType ;
      uLogType: Word;
      uSubType: Word;
      iPrecision: Integer;
      iScale: Integer;
      Length: LongWord ;
      pBuffer: Pointer;
      lInd: Integer): SQLResult;
begin
  Result := I.setParameter(ulParameter, ulChildPos, eParamType, uLogType,
      uSubType, iPrecision, iScale, Length, pBuffer, lInd);
end;

function TISQLCommand25.getParameter(ParameterNumber: Word; ulChildPos: Word; Value: Pointer;
      Length: Integer; var IsBlank: Integer): SQLResult;
begin
  Result := I.getParameter(ParameterNumber, ulChildPos, Value, Length, IsBlank);
end;

function TISQLCommand25.prepare(SQL: PWideChar; ParamCount: Word): SQLResult;
begin
  Result := I.prepare(PChar(AnsiString(SQL)), ParamCount);
end;

function TISQLCommand25.execute(var Cursor: TISQLCursor): SQLResult;
var
  newCursor: ISQLCursor25;
begin
  result := I.execute(newCursor);
  if (newCursor <> Nil) then
    Cursor := TISQLCursor25.Create(newCursor)
  else
    Cursor := Nil;
end;


function TISQLCommand25.executeImmediate(SQL: PWideChar; var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor25;
begin
  Result := I.executeImmediate(PChar(AnsiString(SQL)), iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLCommand25.getNextCursor(var Cursor: TISQLCursor): SQLResult;
var
  newCursor: ISQLCursor25;
begin
  Result := I.getNextCursor(newCursor);
  if (newCursor <> Nil) then
    Cursor := TISQLCursor25.Create(newCursor)
  else
    Cursor := Nil;
end;

function TISQLCommand25.getRowsAffected(var Rows: LongWord): SQLResult;
begin
  Result := I.getRowsAffected(Rows);
end;

function TISQLCommand25.close: SQLResult;
begin
  Result := I.close;
end;

function TISQLCommand25.getErrorMessage(Error: PWideChar): SQLResult;
begin
  Result := I.getErrorMessage(PChar(Error));
end;

function TISQLCommand25.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  Result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLCommand25.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (Result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1));
    Result := I.getErrorMessage(Message);
  end;
  if (Result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

{ TTISQLCursor30 }

constructor TISQLCursor30.Create(NewCursor: ISQLCursor30);
begin
  I := NewCursor;
end;

destructor TISQLCursor30.Destroy;
begin
  I := Nil;
  Inherited;
end;

function TISQLCursor30.SetOption(eOption: TSQLCursorOption;
                     PropValue: LongInt): SQLResult;
begin
  result := I.SetOption(eOption, PropValue);
end;

function TISQLCursor30.SetStringOption(eOption: TSQLCursorOption;
                     const str: WideString): SQLResult;
begin
  result := I.SetOption(eOption, Integer(PWideChar(str)));
end;

function TISQLCursor30.GetOption(eOption: TSQLCursorOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  result := I.GetOption(eOption, PropValue, MaxLength, Length);
end;

function TISQLCursor30.GetStringOption(eOption: TSQLCursorOption;
                     var str: WideString): SQLResult;
var
  Len: SmallInt;
begin
  Len := Length(str) * sizeof(WideChar);
  FillChar(str[1], Len, #0);
  result := I.GetOption(eOption, PWideChar(str), Len * sizeof(widechar), Len);
  SetLength(str, WStrLen(PWideChar(str)))
end;

function TISQLCursor30.getCurObjectTypeName(const iFldNum: Word): WideString;
var
  Len: SmallInt;
  TypeDesc30 : ObjTypeDesc30;
begin
  TypeDesc30.iFldNum := iFldNum;
  if (getOption(eCurObjectTypeName, @TypeDesc30,
    SizeOf(TypeDesc30), Len) = DBXERR_NONE) then
    Result := TypeDesc30.szTypeName
  else
    Result := '';
end;

function TISQLCursor30.getErrorMessage(Error: PWideChar): SQLResult;
begin
  result := I.getErrorMessage(Error);
end;

function TISQLCursor30.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLCursor30.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PWideChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1) * SizeOf(WideChar));
    result := I.getErrorMessage(Message);
  end;
  if (result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

function TISQLCursor30.getColumnCount(var pColumns: Word): SQLResult;
begin
  result := I.getColumnCount(pColumns);
end;

function TISQLCursor30.getColumnNameLength(
      ColumnNumber: Word;
      var pLen: Word): SQLResult;
begin
  result := I.getColumnNameLength(ColumnNumber, pLen);
end;

function TISQLCursor30.getColumnName(ColumnNumber: Word; pColumnName: PWideChar): SQLResult;
begin
  result := I.getColumnName(ColumnNumber, pColumnName);
end;

function TISQLCursor30.getColumnName(ColumnNumber: Word): WideString;
var
  buf : DBINAME128;
begin
  if I.getColumnName(ColumnNumber, buf) = DBXERR_NONE then
    Result := buf
  else
    Result := '';
end;

function TISQLCursor30.getColumnType(ColumnNumber: Word; var puType: Word;
      var puSubType: Word): SQLResult;
begin
  result := I.getColumnType(ColumnNumber, puType, puSubType);
end;

function TISQLCursor30.getColumnLength(ColumnNumber: Word; var pLength: LongWord): SQLResult;
begin
  result := I.getColumnLength(ColumnNumber, pLength);
end;

function TISQLCursor30.getColumnPrecision(ColumnNumber: Word;
      var piPrecision: SmallInt): SQLResult;
begin
  result := I.getColumnPrecision(ColumnNumber, piPrecision);
end;

function TISQLCursor30.getColumnScale(ColumnNumber: Word; var piScale: SmallInt): SQLResult;
begin
  result := I.getColumnScale(ColumnNumber, piScale);
end;

function TISQLCursor30.isNullable(ColumnNumber: Word; var Nullable: LongBool): SQLResult;
begin
  result := I.isNullable(ColumnNumber, Nullable);
end;

function TISQLCursor30.isAutoIncrement(ColumnNumber: Word; var AutoIncr: LongBool): SQLResult;
begin
  result := I.isAutoIncrement(ColumnNumber, AutoIncr);
end;

function TISQLCursor30.isReadOnly(ColumnNumber: Word; var ReadOnly: LongBool): SQLResult;
begin
  result := I.isReadOnly(ColumnNumber, ReadOnly);
end;

function TISQLCursor30.isSearchable(ColumnNumber: Word; var Searchable: LongBool): SQLResult;
begin
  result := I.isSearchable(ColumnNumber, Searchable);
end;

function TISQLCursor30.isBlobSizeExact(ColumnNumber: Word; var IsExact: LongBool): SQLResult;
begin
  result := I.isBlobSizeExact(ColumnNumber, IsExact);
end;

function TISQLCursor30.next: SQLResult;
begin
  result := I.next;
end;

function TISQLCursor30.getString(ColumnNumber: Word; Value: PChar;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getString(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getWideString(ColumnNumber: Word; Value: PWideChar;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getWideString(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getShort(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getShort(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getLong(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getLong(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getInt64(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getInt64(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getDouble(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getDouble(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getBcd(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBcd(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getTimeStamp(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getTimeStamp(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getTime(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getTime(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getDate(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getDate(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getBytes(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBytes(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor30.getBlobSize(ColumnNumber: Word; var Length: LongWord;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBlobSize(ColumnNumber, Length, IsBlank);
end;

function TISQLCursor30.getBlob(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool; Length: LongWord): SQLResult;
begin
  result := I.getBlob(ColumnNumber, Value, IsBlank, Length);
end;


{ TTISQLCursor25 }

constructor TISQLCursor25.Create(NewCursor: ISQLCursor25);
begin
  I := NewCursor;
end;

destructor TISQLCursor25.Destroy;
begin
  I := Nil;
  Inherited;
end;

function TISQLCursor25.SetOption(eOption: TSQLCursorOption;
                     PropValue: LongInt): SQLResult;
begin
  result := I.SetOption(eOption, PropValue);
end;

function TISQLCursor25.SetStringOption(eOption: TSQLCursorOption;
                     const str: WideString): SQLResult;
begin
  result := I.SetOption(eOption, Integer(PChar(AnsiString(str))));
end;

function TISQLCursor25.GetOption(eOption: TSQLCursorOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  result := I.GetOption(eOption, PropValue, MaxLength, Length);
end;

function TISQLCursor25.GetStringOption(eOption: TSQLCursorOption;
                     var str: WideString): SQLResult;
var
  aStr: AnsiString;
  Len, OrgLen: SmallInt;
begin
  OrgLen := Length(str);
  SetLength(aStr, OrgLen * sizeof(widechar));
  FillChar(aStr[1], OrgLen * sizeof(widechar), #0);
  result := I.GetOption(eOption , PAnsiChar(aStr), OrgLen * sizeof(widechar), Len);
  SetLength(aStr, strlen(PChar(aStr)));
  str := Copy(aStr, 1, OrgLen);
end;

function TISQLCursor25.getCurObjectTypeName(const iFldNum: Word): WideString;
var
  Len: SmallInt;
  TypeDesc25 : ObjTypeDesc25;
begin
  TypeDesc25.iFldNum := iFldNum;
  if (getOption(eCurObjectTypeName, @TypeDesc25,
    SizeOf(TypeDesc25), Len) = DBXERR_NONE) then
    Result := TypeDesc25.szTypeName
  else
    Result := '';
end;


function TISQLCursor25.getErrorMessage(Error: PWideChar): SQLResult;
begin
  result := I.getErrorMessage(PChar(Error));
end;

function TISQLCursor25.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLCursor25.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (Result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1));
    Result := I.getErrorMessage(Message);
  end;
  if (Result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

function TISQLCursor25.getColumnCount(var pColumns: Word): SQLResult;
begin
  result := I.getColumnCount(pColumns);
end;

function TISQLCursor25.getColumnNameLength(
      ColumnNumber: Word;
      var pLen: Word): SQLResult;
begin
  result := I.getColumnNameLength(ColumnNumber, pLen);
end;

function TISQLCursor25.getColumnName(ColumnNumber: Word; pColumnName: PWideChar): SQLResult;
var
  buf : DBINAME32;
begin
  result := I.getColumnName(ColumnNumber, buf);
  WStrCopy(pColumnName, PWideChar(WideString(buf)));
end;

function TISQLCursor25.getColumnName(ColumnNumber: Word): WideString;
var
  buf : DBINAME32;
begin
  if I.getColumnName(ColumnNumber, buf) = DBXERR_NONE then
    Result := buf
  else
    Result := '';
end;

function TISQLCursor25.getColumnType(ColumnNumber: Word; var puType: Word;
      var puSubType: Word): SQLResult;
begin
  result := I.getColumnType(ColumnNumber, puType, puSubType);
end;

function TISQLCursor25.getColumnLength(ColumnNumber: Word; var pLength: LongWord): SQLResult;
begin
  result := I.getColumnLength(ColumnNumber, pLength);
end;

function TISQLCursor25.getColumnPrecision(ColumnNumber: Word;
      var piPrecision: SmallInt): SQLResult;
begin
  result := I.getColumnPrecision(ColumnNumber, piPrecision);
end;

function TISQLCursor25.getColumnScale(ColumnNumber: Word; var piScale: SmallInt): SQLResult;
begin
  result := I.getColumnScale(ColumnNumber, piScale);
end;

function TISQLCursor25.isNullable(ColumnNumber: Word; var Nullable: LongBool): SQLResult;
begin
  result := I.isNullable(ColumnNumber, Nullable);
end;

function TISQLCursor25.isAutoIncrement(ColumnNumber: Word; var AutoIncr: LongBool): SQLResult;
begin
  result := I.isAutoIncrement(ColumnNumber, AutoIncr);
end;

function TISQLCursor25.isReadOnly(ColumnNumber: Word; var ReadOnly: LongBool): SQLResult;
begin
  result := I.isReadOnly(ColumnNumber, ReadOnly);
end;

function TISQLCursor25.isSearchable(ColumnNumber: Word; var Searchable: LongBool): SQLResult;
begin
  result := I.isSearchable(ColumnNumber, Searchable);
end;

function TISQLCursor25.isBlobSizeExact(ColumnNumber: Word; var IsExact: LongBool): SQLResult;
begin
  result := I.isBlobSizeExact(ColumnNumber, IsExact);
end;

function TISQLCursor25.next: SQLResult;
begin
  result := I.next;
end;

function TISQLCursor25.getString(ColumnNumber: Word; Value: PChar;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getString(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getWideString(ColumnNumber: Word; Value: PWideChar;
      var IsBlank: LongBool): SQLResult;
begin
  Result := DBXERR_SQLERROR;
end;

function TISQLCursor25.getShort(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getShort(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getLong(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getLong(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getInt64(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  Result := DBXERR_SQLERROR;
end;

function TISQLCursor25.getDouble(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getDouble(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getBcd(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBcd(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getTimeStamp(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getTimeStamp(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getTime(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getTime(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getDate(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getDate(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getBytes(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBytes(ColumnNumber, Value, IsBlank);
end;

function TISQLCursor25.getBlobSize(ColumnNumber: Word; var Length: LongWord;
      var IsBlank: LongBool): SQLResult;
begin
  result := I.getBlobSize(ColumnNumber, Length, IsBlank);
end;

function TISQLCursor25.getBlob(ColumnNumber: Word; Value: Pointer;
      var IsBlank: LongBool; Length: LongWord): SQLResult;
begin
  result := I.getBlob(ColumnNumber, Value, IsBlank, Length);
end;


{ TISQLMetaData30 }

constructor TISQLMetaData30.Create(newMetaData: ISQLMetaData30);
begin
  I := newMetaData;
end;

destructor TISQLMetaData30.Destroy;
begin
  I := Nil;
  inherited;
end;

function TISQLMetaData30.SetOption(eDOption: TSQLMetaDataOption;
                     PropValue: LongInt): SQLResult;
begin
  Result := I.SetOption(eDOption, PropValue);
end;

function TISQLMetaData30.SetStringOption(eDOption: TSQLMetaDataOption;
                     const str: WideString): SQLResult;
begin
  Result := I.SetOption(eDOption, Integer(PWideChar(str)));
end;

function TISQLMetaData30.GetOption(eDOption: TSQLMetaDataOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  Result := I.GetOption(eDOption, PropValue, MaxLength, Length);
end;

function TISQLMetaData30.GetStringOption(eDOption: TSQLMetaDataOption;
                     var str: WideString): SQLResult;
var
  Len: SmallInt;
begin
  Len := Length(str) * sizeof(WideChar);
  FillChar(str[1], Len, #0);
  result := I.GetOption(eDOption, PWideChar(str), Len * sizeof(widechar), Len);
  SetLength(str, WStrLen(PWideChar(str)));
end;

function TISQLMetaData30.getObjectList(eObjType: TSQLObjectType; var Cursor: TISQLCursor):
                     SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getObjectList(eObjType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getTables(TableName: PWideChar; TableType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getTables(TableName, TableType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getProcedures(ProcedureName: PWideChar; ProcType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getProcedures(ProcedureName, ProcType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getColumns(TableName: PWideChar; ColumnName: PWideChar;
                     ColType: LongWord; var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getColumns(TableName, ColumnName, ColType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getProcedureParams(ProcName: PWideChar; ParamName: PWideChar;
                     var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getProcedureParams(ProcName, ParamName, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getIndices(TableName: PWideChar; IndexType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  iCursor: ISQLCursor30;
begin
  Result := I.getIndices(TableName, IndexType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor30.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData30.getErrorMessage(Error: PWideChar): SQLResult;
begin
  Result := I.getErrorMessage(Error);
end;

function TISQLMetaData30.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  Result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLMetaData30.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PWideChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1) * SizeOf(WideChar));
    result := I.getErrorMessage(Message);
  end;
  if (result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;


{ TISQLMetaData25 }

constructor TISQLMetaData25.Create(newMetaData: ISQLMetaData25);
begin
  I := newMetaData;
end;

destructor TISQLMetaData25.Destroy;
begin
  I := Nil;
  inherited;
end;

function TISQLMetaData25.SetOption(eDOption: TSQLMetaDataOption;
                     PropValue: LongInt): SQLResult;
begin
  Result := I.SetOption(eDOption, PropValue);
end;

function TISQLMetaData25.SetStringOption(eDOption: TSQLMetaDataOption;
                     const str: WideString): SQLResult;
begin
  Result := I.SetOption(eDOption, Integer(PChar(AnsiString(str))));
end;

function TISQLMetaData25.GetOption(eDOption: TSQLMetaDataOption; PropValue: Pointer;
                     MaxLength: SmallInt; out Length: SmallInt): SQLResult;
begin
  Result := I.GetOption(eDOption, PropValue, MaxLength, Length);
end;

function TISQLMetaData25.GetStringOption(eDOption: TSQLMetaDataOption;
                     var str: WideString): SQLResult;
var
  aStr: AnsiString;
  Len, OrgLen: SmallInt;
begin
  OrgLen := Length(str);
  SetLength(aStr, OrgLen * sizeof(widechar));
  FillChar(aStr[1], OrgLen * sizeof(widechar), #0);
  result := I.GetOption(eDOption, PAnsiChar(aStr), OrgLen * sizeof(widechar), Len);
  SetLength(aStr, strlen(PChar(aStr)));
  str := Copy(aStr, 1, OrgLen);
end;

function TISQLMetaData25.getObjectList(eObjType: TSQLObjectType; var Cursor: TISQLCursor):
                     SQLResult;
var
  iCursor: ISQLCursor25;
begin
  Result := I.getObjectList(eObjType, iCursor);
  if (iCursor <> Nil)then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getTables(TableName: PWideChar; TableType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  aTableName: DBINAME32;
  iCursor: ISQLCursor25;
begin
  StrLCopy(aTableName, PAnsiChar(AnsiString(TableName)), 32);
  Result := I.getTables(aTableName, TableType, iCursor);
  if (iCursor <> Nil)then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getProcedures(ProcedureName: PWideChar; ProcType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  aProcedureName: DBINAME32;
  iCursor: ISQLCursor25;
begin
  StrLCopy(aProcedureName, PAnsiChar(AnsiString(ProcedureName )), 32);
  Result := I.getProcedures(aProcedureName, ProcType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getColumns(TableName: PWideChar; ColumnName: PWideChar;
                     ColType: LongWord; var Cursor: TISQLCursor): SQLResult;
var
  aTableName, aColumnName: DBINAME32;
  iCursor: ISQLCursor25;
begin
  StrLCopy(aTableName , PAnsiChar(AnsiString(TableName)), 32);
  StrLCopy(aColumnName , PAnsiChar(AnsiString(ColumnName)), 32);
  Result := I.getColumns(aTableName, aColumnName, ColType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getProcedureParams(ProcName: PWideChar; ParamName: PWideChar;
                     var Cursor: TISQLCursor): SQLResult;
var
  aProcName, aParamName: DBINAME32;
  iCursor: ISQLCursor25;
begin
  StrLCopy(aProcName, PAnsiChar(AnsiString(ProcName )), 32);
  StrLCopy(aParamName, PAnsiChar(AnsiString(ParamName)), 32);
  Result := I.getProcedureParams(aProcName, aParamName, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getIndices(TableName: PWideChar; IndexType: LongWord;
                     var Cursor: TISQLCursor): SQLResult;
var
  aTableName: DBINAME32;
  iCursor: ISQLCursor25;
begin
  StrLCopy(aTableName , PAnsiChar(AnsiString(TableName)), 32);
  Result := I.getIndices(aTableName, IndexType, iCursor);
  if (iCursor <> Nil) then
    Cursor := TISQLCursor25.Create(iCursor)
  else
    Cursor := Nil;
end;

function TISQLMetaData25.getErrorMessage(Error: PWideChar): SQLResult;
begin
  Result := I.getErrorMessage(PAnsiChar(Error));
end;

function TISQLMetaData25.getErrorMessageLen(out ErrorLen: SmallInt): SQLResult;
begin
  Result := I.getErrorMessageLen(ErrorLen);
end;

function TISQLMetaData25.getErrorMessage(var Error: WideString): SQLResult;
var
  Message: PChar;
  MessageLen: SmallInt;
begin
  Message := nil;
  Error := '';
  result := I.getErrorMessageLen(MessageLen);
  if (Result = DBXERR_NONE) and (MessageLen > 0) then
  begin
    Message := AllocMem((MessageLen + 1));
    Result := I.getErrorMessage(Message);
  end;
  if (Result = DBXERR_NONE) then
    Error := Message;
  if Assigned(Message) then
    FreeMem(Message);
end;

{ TFLDDesc30 }

function TFLDDesc30.GetiFldNum: Word;
begin
  Result := FFLDDesc30.iFldNum;
end;

function TFLDDesc30.GetszName: WideString;
begin
  Result := FFLDDesc30.szName;
end;

function TFLDDesc30.GetiFldType: Word;
begin
  Result := FFLDDesc30.iFldType;
end;

function TFLDDesc30.GetiSubType: Word;
begin
  Result := FFLDDesc30.iSubType;
end;

function TFLDDesc30.GetiUnits1: SmallInt;
begin
  Result := FFLDDesc30.iUnits1;
end;

function TFLDDesc30.GetiUnits2: SmallInt;
begin
  Result := FFLDDesc30.iUnits2;
end;

function TFLDDesc30.GetiOffset: Word;
begin
  Result := FFLDDesc30.iOffset;
end;

function TFLDDesc30.GetiLen: LongWord;
begin
  Result := FFLDDesc30.iLen;
end;

function TFLDDesc30.GetiNullOffset: Word;
begin
  Result := FFLDDesc30.iNullOffset;
end;

function TFLDDesc30.GetefldvVchk: FLDVchk;
begin
  Result := FFLDDesc30.efldvVchk;
end;

function TFLDDesc30.GetefldrRights: FLDRights;
begin
  Result := FFLDDesc30.efldrRights;
end;

function TFLDDesc30.GetbCalcField: WordBool;
begin
  Result := FFLDDesc30.bCalcField;
end;

procedure TFLDDesc30.SetiFldNum(Value: Word);
begin
  FFLDDesc30.iFldNum := Value;
end;

procedure TFLDDesc30.SetszName(Value: WideString);
begin
  WStrLCopy(FFLDDesc30.szName, PWideChar(Value), sizeof(FFLDDesc30.szName) div sizeof(widechar));
end;

procedure TFLDDesc30.SetiFldType(Value: Word);
begin
  FFLDDesc30.iFldType  := Value;
end;

procedure TFLDDesc30.SetiSubType(Value: Word);
begin
  FFLDDesc30.iSubType := Value;
end;

procedure TFLDDesc30.SetiUnits1(Value: SmallInt);
begin
  FFLDDesc30.iUnits1 := Value;
end;

procedure TFLDDesc30.SetiUnits2(Value: SmallInt);
begin
  FFLDDesc30.iUnits2 := Value;
end;

procedure TFLDDesc30.SetiOffset(Value: Word);
begin
  FFLDDesc30.iOffset := Value;
end;

procedure TFLDDesc30.SetiLen(Value: LongWord);
begin
  FFLDDesc30.iLen := Value;
end;

procedure TFLDDesc30.SetiNullOffset(Value: Word);
begin
  FFLDDesc30.iNullOffset := Value;
end;

procedure TFLDDesc30.SetefldvVchk(Value: FLDVchk);
begin
  FFLDDesc30.efldvVchk := Value;
end;

procedure TFLDDesc30.SetefldrRights(Value: FLDRights);
begin
  FFLDDesc30.efldrRights := Value;
end;

procedure TFLDDesc30.SetbCalcField(Value: WordBool);
begin
  FFLDDesc30.bCalcField := Value;
end;

{ TFLDDesc25 }

function TFLDDesc25.GetiFldNum: Word;
begin
  Result := FFLDDesc25.iFldNum;
end;

function TFLDDesc25.GetszName: WideString;
begin
//  Result := AnsiString(FFLDDesc25.szName);
  Result := FFLDDesc25.szName;
end;

function TFLDDesc25.GetiFldType: Word;
begin
  Result := FFLDDesc25.iFldType;
end;

function TFLDDesc25.GetiSubType: Word;
begin
  Result := FFLDDesc25.iSubType;
end;

function TFLDDesc25.GetiUnits1: SmallInt;
begin
  Result := FFLDDesc25.iUnits1;
end;

function TFLDDesc25.GetiUnits2: SmallInt;
begin
  Result := FFLDDesc25.iUnits2;
end;

function TFLDDesc25.GetiOffset: Word;
begin
  Result := FFLDDesc25.iOffset;
end;

function TFLDDesc25.GetiLen: LongWord;
begin
  Result := FFLDDesc25.iLen;
end;

function TFLDDesc25.GetiNullOffset: Word;
begin
  Result := FFLDDesc25.iNullOffset;
end;

function TFLDDesc25.GetefldvVchk: FLDVchk;
begin
  Result := FFLDDesc25.efldvVchk;
end;

function TFLDDesc25.GetefldrRights: FLDRights;
begin
  Result := FFLDDesc25.efldrRights;
end;

function TFLDDesc25.GetbCalcField: WordBool;
begin
  Result := FFLDDesc25.bCalcField;
end;

procedure TFLDDesc25.SetiFldNum(Value: Word);
begin
  FFLDDesc25.iFldNum := Value;
end;

procedure TFLDDesc25.SetszName(Value: WideString);
var
  aStr: AnsiString;
begin
  aStr := Value;
  StrLCopy(FFLDDesc25.szName, PAnsiChar(aStr), sizeof(FFLDDesc25.szName));
end;

procedure TFLDDesc25.SetiFldType(Value: Word);
begin
  FFLDDesc25.iFldType  := Value;
end;

procedure TFLDDesc25.SetiSubType(Value: Word);
begin
  FFLDDesc25.iSubType := Value;
end;

procedure TFLDDesc25.SetiUnits1(Value: SmallInt);
begin
  FFLDDesc25.iUnits1 := Value;
end;

procedure TFLDDesc25.SetiUnits2(Value: SmallInt);
begin
  FFLDDesc25.iUnits2 := Value;
end;

procedure TFLDDesc25.SetiOffset(Value: Word);
begin
  FFLDDesc25.iOffset := Value;
end;

procedure TFLDDesc25.SetiLen(Value: LongWord);
begin
  FFLDDesc25.iLen := Value;
end;

procedure TFLDDesc25.SetiNullOffset(Value: Word);
begin
  FFLDDesc25.iNullOffset := Value;
end;

procedure TFLDDesc25.SetefldvVchk(Value: FLDVchk);
begin
  FFLDDesc25.efldvVchk := Value;
end;

procedure TFLDDesc25.SetefldrRights(Value: FLDRights);
begin
  FFLDDesc25.efldrRights := Value;
end;

procedure TFLDDesc25.SetbCalcField(Value: WordBool);
begin
  FFLDDesc25.bCalcField := Value;
end;

{ TSQLParams }

type

{ TSQLParams }

  TSQLParams = class(TParams)
  private
    FFieldName: TWideStrings;
    FBindAllFields: Boolean;
    function ParseSelect(SQL: WideString; bDeleteQuery: Boolean): WideString;
    function ParseUpdate(SQL: WideString): WideString;
    function ParseInsert(SQL: WideString): WideString;
  public
    constructor Create(Owner: TPersistent);
    Destructor Destroy; override;
    function GetFieldName(index: Integer): WideString;
    function Parse(Var SQL: WideString; DoCreate: Boolean): WideString;
    property BindAllFields: Boolean read FBindAllFields;
  end;

constructor TSQLParams.Create(Owner: TPersistent);
begin
  inherited;
  FBindAllFields := False;
  FFieldName := TWideStringList.Create;
end;

destructor TSQLParams.Destroy;
begin
  inherited;
  FreeAndNil(FFieldName);
end;

function TSQLParams.GetFieldName(index: Integer): WideString;
begin
   Result := FFieldName[ index ];
end;

function TSQLParams.Parse(var SQL: WideString; DoCreate: Boolean): WideString;
const
  SDelete = 'delete';      { Do not localize }
  SUpdate = 'update';      { Do not localize }
  SInsert = 'insert';      { Do not localize }
var
  Start: string;
begin
  SQL := ParseSQL(SQL, DoCreate);
  Start := WideLowerCase(copy(SQL, 1, 6));
{ attempt to determine fields and fieldtypes associated with params }
  if Start = SSelect then
    Result := ParseSelect(SQL, False)
  else if Start = SDelete then
    Result := ParseSelect(SQL, True)
  else if Start = SInsert then
    Result := ParseInsert(SQL)
  else if Start = SUpdate then
    Result := ParseUpdate(SQL)
  else
    Result := '';
end;

{ no attempt to match fields clause with values clause :
    types only added if all values are inserted }
function TSQLParams.ParseInsert(SQL: WideString): WideString;
var
  Start: PWideChar;
  Value: Widestring;
  CurSection: TSQLToken;
begin
  Result := '';
  if ((Owner <> nil) and (Owner is TCustomSqlDataSet) and Assigned(TCustomSqlDataSet(Owner).ParseInsertSql)) then
    TCustomSqlDataSet(Owner).ParseInsertSql(FFieldName, SQL, FBindAllFields, Result)
  else
  begin
    if Pos(SSelectSpaces, WideLowerCase(SQL)) > 1 then Exit;  // can't parse sub queries
    Start := PWideChar(SQL);
    CurSection := stUnknown;
    { move past 'insert ' }
    NextSQLToken(Start, Value, CurSection);
    { move past 'into ' }
    NextSQLToken(Start, Value, CurSection);
    { move past <TableName> }
    NextSQLToken(Start, Value, CurSection);
  
    { Check for owner qualified table name }
    if Start^ = '.' then
      NextSQLToken(Start, Value, CurSection);
    Result := Value;
  
    { move past 'set' }
    NextSQLToken(Start, Value, CurSection);
    if (WideLowerCase(Value) = 'values') then
      FBindAllFields := True;
  end;
end;

function TSQLParams.ParseSelect(SQL: WideString; bDeleteQuery: Boolean): WideString;
var
  bParsed: Boolean;
  FWhereFound: Boolean;
  Start: PWideChar;
  FName, Value: WideString;
  SQLToken, CurSection, LastToken: TSQLToken;
  Params: Integer;
begin
  Result := '';
  bParsed := False;
  if ((Owner <> nil) and (Owner is TCustomSqlDataSet)) then
  begin
    if (not bDeleteQuery) and Assigned(TCustomSqlDataSet(Owner).ParseSelectSql) then
    begin
      TCustomSqlDataSet(Owner).ParseSelectSql(FFieldName, SQL, Result);
      bParsed := True;
    end else if bDeleteQuery and Assigned(TCustomSqlDataSet(Owner).ParseDeleteSql) then
    begin
      TCustomSqlDataSet(Owner).ParseDeleteSql(FFieldName, SQL, Result);
      bParsed := True;
    end;
  end;
  if not bParsed then
  begin
    if bDeleteQuery = False then
    begin
      if Pos(SSelectSpaces, WideLowerCase(Widestring(PWideChar(SQL)+8))) > 1 then Exit;  // can't parse sub queries
      Start := PWideChar(SQL);
    end else
    begin
      if Pos(SSelectSpaces, WideLowerCase(SQL)) > 1 then Exit;  // can't parse sub queries
      Start := PWideChar(SSelectStar + Copy(SQL, 8, Length(SQL) -7));
    end;
    CurSection := stUnknown;
    LastToken := stUnknown;
    FWhereFound := False;
    Params := 0;
    repeat
      repeat
        SQLToken := NextSQLToken(Start, FName, CurSection);
        if SQLToken in [stWhere] then
        begin
          FWhereFound := True;
          LastToken := stWhere;
        end else if SQLToken in [stTableName] then
        begin
          { Check for owner qualified table name }
          if Start^ = '.' then
            NextSQLToken(Start, FName, CurSection);
          Result := FName;
        end else
        if (SQLToken = stValue) and (LastToken = stWhere) then
          SQLToken := stFieldName;
        if SQLToken in SQLSections then CurSection := SQLToken;
      until SQLToken in [stFieldName, stEnd];
      if FWhereFound and (SQLToken in [stFieldName]) then
        repeat
          SQLToken := NextSQLToken(Start, Value, CurSection);
            if SQLToken in SQLSections then CurSection := SQLToken;
        until SQLToken in [stEnd,stValue,stIsNull,stIsNotNull,stFieldName];
      if Value='?' then
      begin
       FFieldName.Add(FName);
        Inc(Params);
      end;
    until (Params = Count) or (SQLToken in [stEnd]);
    if Result = '' then Result := GetTableNameFromSql(SQL);
  end;
end;

function TSQLParams.ParseUpdate(SQL: WideString): WideString;
var
  Start: PWideChar;
  FName, Value: WideString;
  SQLToken, CurSection: TSQLToken;
  Params: Integer;
begin
  Result := '';
  if ((Owner <> nil) and (Owner is TCustomSqlDataSet) and Assigned(TCustomSqlDataSet(Owner).ParseUpdateSql)) then
    TCustomSqlDataSet(Owner).ParseUpdateSql(FFieldName, SQL, Result)
  else
  begin
    if Pos(SSelectSpaces, WideLowerCase(SQL)) > 1 then Exit;  // can't parse sub queries
    Start := PWideChar(SQL);
    CurSection := stUnknown;
    { move past 'update ' }
    NextSQLToken(Start, FName, CurSection);
    { move past <TableName> }
    NextSQLToken(Start, FName, CurSection);

    { Check for owner qualified table name }
    if Start^ = '.' then
      NextSQLToken(Start, FName, CurSection);

    Result := FName;
    { move past 'set ' }
    NextSQLToken(Start, FName, CurSection);
    Params := 0;
    CurSection := stSelect;
    repeat
      repeat
        SQLToken := NextSQLToken(Start, FName, CurSection);
        if SQLToken in SQLSections then CurSection := SQLToken;
      until SQLToken in [stFieldName, stEnd];
      if Pos(WideLowerCase(FName), 'values(') > 0 then continue;   { do not localize }
      if Pos(WideLowerCase(FName), 'values (') > 0 then continue;  { do not localize }
      if SQLToken in [stFieldName] then
        repeat
          SQLToken := NextSQLToken(Start, Value, CurSection);
            if SQLToken in SQLSections then CurSection := SQLToken;
        until SQLToken in [stEnd,stValue,stIsNull,stIsNotNull,stFieldName];
      if Value='?' then
      begin
        FFieldName.Add(FName);
        Inc(Params);
      end;
    until (Params = Count) or (SQLToken in [stEnd]);
  end;
end;

{ TSQLMonitor }

constructor TSQLMonitor.Create(AOwner: TComponent);
begin
  FTraceList := TWideStringList.Create;
  FMaxTraceCount := -1;
  inherited;
end;

destructor TSQLMonitor.Destroy;
begin
  if Active then SetActive(False);
  SetSQLConnection(nil);
  inherited;
  FreeAndNil(FTraceList);
end;

procedure TSQLMonitor.SetFileName(const Value: String);
begin
  FFileName := Value;
end;

procedure TSQLMonitor.CheckInactive;
begin
  if FActive then
  begin
    if (csDesigning in ComponentState) or (csLoading in ComponentState) then
      SetActive(False)
    else
      DatabaseError(SMonitorActive, Self);
  end;
end;

procedure TSQLMonitor.SetSQLConnection(Value: TSQLConnection);
var
  IsActive: Boolean;
begin
  if Value <> FSQLConnection then
  begin
    IsActive := Active;
    CheckInactive;
    if Assigned(FSQLConnection) and not FKeepConnection then
      SQLConnection.UnregisterTraceMonitor(Self);
    FSQLConnection := Value;
    if Assigned(FSQLConnection) then
    begin
      FSQLConnection.RegisterTraceMonitor(Self);
      Active := IsActive;
    end;
  end;
end;

procedure TSQLMonitor.SwitchConnection(const Value: TSQLConnection);
var
  MonitorActive: Boolean;
begin
  FKeepConnection := True;
  MonitorActive := Active;
  if MonitorActive then
    SetActive(False);
  SQLConnection := Value;
  if MonitorActive and Assigned(Value) then
    SetActive(True);
  FKeepConnection := False;
end;

procedure TSQLMonitor.Trace(Desc: pSQLTraceDesc; LogTrace: Boolean);
begin
  if Assigned(FOnTrace) then
    FOnTrace(Self, Desc, LogTrace);
end;

function TSQLMonitor.InvokeCallBack(CallType: TRACECat; CBInfo: Pointer): CBRType; stdcall;
var
  Desc: pSQLTraceDesc;
  LogTrace: Boolean;
  Msg: WideString;
begin
  Result := cbrUSEDEF;
  if csDestroying in ComponentState then exit;
  Desc := pSQLTraceDesc(CBInfo);
  LogTrace := (TSQLTraceFlag(CallType) in FTraceFlags) or (FTraceFlags = []);
  Trace(Desc, LogTrace);
  if LogTrace then
  begin
    SetLength(Msg, WStrLen(Desc.pszTrace));
    Move(Desc.pszTrace[0], PWideChar(Msg)[0], WStrLen(Desc.pszTrace) * 2);                
    if (FMaxTraceCount = -1) or (TraceCount < FMaxTraceCount) then
      FTraceList.Add(Msg);
    if Assigned(FOnLogTrace) then
      FOnLogTrace(Self, Desc);
    if FAutoSave and (FFileName <> '') then
      SaveToFile('');
  end;
end;

function SQLCallBack(CallType: TRACECat; CBInfo: Pointer): CBRType; stdcall;
begin
  Result := cbrUSEDEF;
  if CBInfo <> nil then
    Result := TSQLMonitor(PSQLTraceDesc(CBInfo).ClientData).InvokeCallback(CallType, CBInfo);
end;

function SQLCallBack25(CallType: TRACECat; CBInfo25: Pointer): CBRType; stdcall;
var
  CBInfo30 : SQLTraceDesc30;
begin
  Result := cbrUSEDEF;
  if CBInfo25 <> nil then
  begin
    WStrPLCopy(CBInfo30.pszTrace, pSQLTRACEDesc25(CBInfo25).pszTrace, 1023);
    CBInfo30.eTraceCat := pSQLTRACEDesc30(CBInfo25).eTraceCat;
    CBInfo30.ClientData   := pSQLTRACEDesc30(CBInfo25).ClientData;
    CBInfo30.uTotalMsgLen := pSQLTRACEDesc30(CBInfo25).uTotalMsgLen;
    Result := TSQLMonitor(pSQLTRACEDesc25(CBInfo25).ClientData).InvokeCallback(CallType, @CBInfo30);
  end;
end;

procedure TSQLMonitor.UpdateTraceCallBack;
begin
  if Assigned(FSQLConnection) then
  begin
    if Assigned(FSQLConnection.SQLConnection) then
    begin
      if FSQLConnection.SQLConnection is TISQLConnection25 then
        FSQLConnection.SetTraceCallbackEvent(SQLCallBack25, Integer(Self))
      else
        FSQLConnection.SetTraceCallbackEvent(SQLCallBack, Integer(Self));
    end
    else
      FSQLConnection.SetTraceCallbackEvent(nil, Integer(0));
  end;
end;

procedure TSQLMonitor.SetActive(Value: Boolean);
var
  FileHandle: Integer;
begin
  if FActive <> Value then
  begin
    if (csReading in ComponentState) then
      FStreamedActive := Value
    else begin
      if not (csDestroying in ComponentState) and not Assigned(FSQLConnection) then
        DatabaseError(SConnectionNameMissing)
      else
      begin
        if Value and (FileName <> '') then
        begin
          if not FileExists(FileName) then
          begin
            FileHandle := FileCreate(FileName);
            if FileHandle = -1 then
              DatabaseErrorFmt(SCannotCreateFile, [FileName])
            else
              FileClose(FileHandle);
          end;
        end;
        if Assigned(FSQLConnection) then
        begin
          if Value then
            UpdateTraceCallBack
          else
            FSQLConnection.SetTraceCallbackEvent(nil, Integer(0));
        end;
        FActive := Value;
      end;
    end;
  end;
end;

procedure TSQLMonitor.SetStreamedActive;
begin
  if FStreamedActive then
    SetActive(True);
end;

function TSQLMonitor.GetTraceCount: Integer;
begin
  Result := FTraceList.Count;
end;

procedure TSQLMonitor.LoadFromFile(AFileName: string);
begin
  if AFileName <> '' then
    FTraceList.LoadFromFile(AFileName)
  else if FFileName <> '' then
    FTraceList.LoadFromFile(string(FFileName))
  else
    DatabaseError(SFileNameBlank);
end;

procedure TSQLMonitor.SaveToFile(AFileName: string);
begin
  if AFileName <> '' then
    FTraceList.SaveToFile(AFileName)
  else if FFileName <> '' then
    FTraceList.SaveToFile(FFileName)
  else
    DatabaseError(SFileNameBlank);
end;

procedure TSQLMonitor.SetTraceList(Value: TWideStrings);
begin
  if FTraceList <> Value then
  begin
    FTraceList.BeginUpdate;
    try
      FTraceList.Assign(Value);
    finally
      FTraceList.EndUpdate;
    end;
  end;
end;


{ TSQLConnection }

constructor TSQLConnection.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FParams := TWideStringList.Create;
  FAutoClone := True;
  try
    FConnectionRegistryFile := GetConnectionRegistryFile(csDesigning in ComponentState);
  except
    FConnectionRegistryFile := '';
  end;
  FKeepConnection := True;
  FMonitorUsers := TList.Create;
  FSQLHourGlass := True;
  FQuoteChar := '';
  FTableScope := [tsTable, tsView];
  LoginPrompt := True;
  FLoginUserName := '';
  FISQLConnection := Nil;                                  
end;

destructor TSQLConnection.Destroy;
begin
  Destroying;
  ClearConnectionUsers;
  Close;
  ClearMonitors;
  FreeAndNil(FMonitorUsers);
  if Assigned(FISQLConnection) then
    FreeAndNil(FISQLConnection);
  inherited Destroy;
  FreeAndNil(FParams);
end;

{ user registration }

procedure TSQLConnection.ClearConnectionUsers;
begin
  while DataSetCount > 0 do
  begin
    if TCustomSQLDataSet(DataSets[0]).Active then
      TCustomSQLDataSet(DataSets[0]).Close;
    TCustomSQLDataSet(DataSets[0]).FreeStatement;
    TCustomSQLDataSet(DataSets[0]).SetConnection(nil);
  end;
end;

procedure TSQLConnection.ClearMonitors;
var
  I: Integer;
begin
  for I := 0 to FMonitorUsers.Count -1 do
  begin
    if Self.FIsCloned then
      TSQLMonitor(FMonitorUsers[I]).SwitchConnection(Self.FCloneParent)
    else
    begin
      TSQLMonitor(FMonitorUsers[I]).SetActive(False);
      TSQLMonitor(FMonitorUsers[I]).FSQLConnection := nil;
    end;
  end;
end;

procedure TSQLConnection.RegisterTraceMonitor(Client: TObject);
begin
  FMonitorUsers.Add(Client);
end;

procedure TSQLConnection.UnregisterTraceMonitor(Client: TObject);
begin
  FMonitorUsers.Remove(Client);
end;

{ Driver Exception handling routine }
const
  DbxError : array[0..28] of String = (SqlConst.SNOERROR, SqlConst.SWARNING,
      SqlConst.SNOMEMORY, SqlConst.SINVALIDFLDTYPE, SqlConst.SINVALIDHNDL,
      SqlConst.SNOTSUPPORTED, SqlConst.SINVALIDTIME, SqlConst.SINVALIDXLATION,
      SqlConst.SOUTOFRANGE, SqlConst.SINVALIDPARAM, SqlConst.SEOF,
      SqlConst.SSQLPARAMNOTSET, SqlConst.SINVALIDUSRPASS, SqlConst.SINVALIDPRECISION,
      SqlConst.SINVALIDLEN, SqlConst.SINVALIDXISOLEVEL, SqlConst.SINVALIDTXNID,
      SqlConst.SDUPLICATETXNID, SqlConst.SDRIVERRESTRICTED, SqlConst.SLOCALTRANSACTIVE,
      SqlConst.SMULTIPLETRANSNOTENABLED, SqlConst.SCONNECTIONFAILED,
      SqlConst.SDRIVERINITFAILED, SqlConst.SOPTLOCKFAILED, SqlConst.SINVALIDREF,
      SqlConst.SNOTABLE, SqlConst.SMISSINGPARAMINSQL, SqlConst.SNOTIMPLEMENTED,
      SqlConst.SDRIVERINCOMPATIBLE);


procedure TSQLConnection.SQLError(OpStatus: SQLResult; eType: TSQLExceptionType; const Command: TISQLCommand = nil);
var
  dbxErrorMsg, ServerErrorMsg, ExceptionMessage: string;
  ServerMessage: WideString;
  Status: SQLResult;
begin
  dbxErrorMsg := '';
  ServerErrorMsg := '';
  ExceptionMessage := '';
  Status := SQL_NULL_DATA;
  if (OpStatus > 0) and (OpStatus <=  DBX_MAXSTATICERRORS) then
  begin
    if OpStatus = 64 then dbxErrorMsg := Format(SDBXError, [SqlConst.SNODATA])
    else if OpStatus = 65 then dbxErrorMsg := Format(SDBXError, [SqlConst.SSQLERROR])
    else dbxErrorMsg := Format(SDBXError, [DbxError[OpStatus]]);
  end;

  case eType of
    exceptCommand:
      Status := Command.getErrorMessage(ServerMessage);
    exceptConnection:
      Status := FISQLConnection.getErrorMessage(ServerMessage);
    exceptMetaData:
      Status := FSQLMetaData.getErrorMessage(ServerMessage);
  end;


  if Status = DBXERR_NONE then
    if Length(ServerMessage) > 0 then
      ServerErrorMsg := WideFormat(SSQLServerError, [ServerMessage]);

  if Length(dbxErrorMsg) > 0 then
    ExceptionMessage := dbxErrorMsg;
  if Length(ServerErrorMsg) > 0 then
  begin
    if Length(ExceptionMessage) > 0 then
      ExceptionMessage := ExceptionMessage + #13 + #10;
    ExceptionMessage := ExceptionMessage + ServerErrorMsg;
  end;
  if (Length(ExceptionMessage) = 0) and (LastError <> '') then
    ExceptionMessage := LastError;
  if Length(ExceptionMessage) = 0 then
    ExceptionMessage :=  Format(SDBXUNKNOWNERROR, [intToStr(OpStatus)]);
  FLastError := ExceptionMessage;
  DatabaseError(ExceptionMessage);
end;

{ loading, connecting and disconnecting }

procedure TSQLConnection.LoadSQLDll;
begin
{$IFDEF MSWINDOWS}
  if DllHandle = THandle(0) then
  begin
{$ENDIF}
  try
    SetCursor(HourGlassCursor);
    if SQLDllHandle = THandle(0) then
      SQLDllHandle := THandle(LoadLibrary(PChar(trim(LibraryName))));
    if SQLDllHandle = THandle(0) then
      DataBaseErrorFmt(sDLLLoadError, [LibraryName]);
    GetDriver := GetProcAddress(HMODULE(SQLDllHandle), PChar(trim(GetDriverFunc)));
    if not Assigned(GetDriver) then
      DataBaseErrorFmt(sDLLProcLoadError, [GetDriverFunc])
  finally
    SetCursor(DefaultCursor);
  end;
{$IFDEF MSWINDOWS}
end;
{$ENDIF}
end;

procedure TSQLConnection.CheckConnection(eFlag: eConnectFlag);
begin
  if (eFlag in [eDisconnect, eReconnect]) then
    Close;
  if (eFlag in [eConnect, eReconnect]) then
    Open
end;

procedure TSQLConnection.Login(LoginParams: TWideStrings);
var
  UserName, Password: string;

  function Login: Boolean;
  begin
    Result := Assigned(FOnLogin);
    if Result then FOnLogin(Self, LoginParams);
  end;

begin
  if not Login then
  begin
    UserName := LoginParams.Values[szUserName];
    if Assigned(LoginDialogExProc) then
    begin
      SetCursor(DefaultCursor);
      if not LoginDialogExProc(ConnectionName, UserName, Password, False) then
        DatabaseErrorFmt(SLoginError, [ConnectionName]);
      SetCursor(HourGlassCursor);
      LoginParams.Values[szUSERNAME] := UserName;
      LoginParams.Values[szPASSWORD] := Password;
    end;
  end;
end;

procedure TSQLConnection.CheckLoginParams;
var
  I: Integer;
begin
  if FLoadParamsOnConnect then
  begin
    LoadParamsFromIniFile;
    FDriverName := GetProfileString(FConnectionName, DRIVERNAME_KEY, ConnectionRegistryFile);
  end;
  if FDriverName = '' then DataBaseError(SMissingDriverName);
  if LoadParamsOnConnect then
    FLibraryName := Trim(GetProfileString(FDriverName, DLLLIB_KEY, GetDriverRegistryFile(csDesigning in ComponentState)));
  if FLibraryName = '' then DataBaseError(SMissingDLLName, Self);
  if LoadParamsOnConnect then
    FVendorLib := Trim(GetProfileString(FDriverName, VENDORLIB_KEY, GetDriverRegistryFile));
  if FVendorLib = '' then DataBaseError(SMissingDLLName, Self);
  if LoadParamsOnConnect then
    FGetDriverFunc := Trim(GetProfileString(FDriverName, GETDRIVERFUNC_KEY, GetDriverRegistryFile));
  if Params.Values[DATABASENAME_KEY] = '' then
  begin
    if FConnectionName = '' then DataBaseError(SConnectionNameMissing)
    else DataBaseError(SMissingDatabaseName);
  end;
  for I := 0 to FMonitorUsers.Count -1 do
    TSQLMonitor(FMonitorUsers[i]).SetStreamedActive;
end;

function TSQLConnection.GetQuoteChar: WideString;
var
  Status: SQLResult;
  str: WideString;
begin
  FQuoteChar := '';
  SetLength(str, 2);
  Status := FSQLMetadata.GetStringOption(eMetaObjectQuoteChar, str);
  if (str <> '') and (Status = DBXERR_NONE) then
    FQuoteChar := str;
  Result := FQuoteChar;
end;

procedure TSQLConnection.SetCursor(CursorType: Integer);
begin
  if SQLHourGlass or (CursorType = DefaultCursor) then
    if Assigned(ScreenCursorProc) then
      ScreenCursorProc(CursorType);
end;

procedure TSQLConnection.ConnectionOptions;
var
  PropSize: SmallInt;
  SupTransactions: LongBool;
begin
  GetQuoteChar;
  if FParams.Values[MAXBLOBSIZE_KEY] <> '' then
    FISQLConnection.SetOption(eConnBlobSize, LongInt(StrToInt(trim(FParams.Values[MAXBLOBSIZE_KEY]))))
  else
    FISQLConnection.SetOption(eConnBlobSize, LongInt(DefaultMaxBlobSize));
  FSQLMetaData.GetOption(eMetaSupportsTransaction, @SupTransactions, SizeOf(Integer), PropSize);
  if SupTransactions then
    FTransActionsSupported := True
  else
    FTransActionsSupported := False;
  FSQLMetaData.GetOption(eMetaSupportsTransactions, @FSupportsMultiTrans, SizeOf(Integer), PropSize);
end;

type
  TISQLConnectionRef = class of TISQLConnection;
  TDBXDrvVersion = record
    DrvVersion: string;
    ProdVersion: string;
    SQLConnection: TISQLConnectionRef;
  end;

const
  ProductVersionStr = '3.0';
  MaxDBXDrvTableEntry = 1;
  DBXDrvMap : array[1..MaxDBXDrvTableEntry] of TDBXDrvVersion = (
   (DrvVersion: DBXDRIVERVERSION30; ProdVersion: DBXPRODUCTVERSION30; SQLConnection:TISQLConnection30)
  );

procedure TSQLConnection.DoConnect;
var
  Status: SQLResult;
  LoginParams: TWideStrings;
  PropSize: SmallInt;
  TrimmedUserName: WideString;
  DrvVersionStr: string;
  str: WideString;
  ind: integer;
  ConnectionStr: WideString;
  SQLConnection: TISQLConnectionRef;
  NewSQLConnection: ISQLConnection;
begin
  CheckLoginParams;
  ConnectionState := csStateConnecting;
  LoadSQLDll;
  LoginParams := TWideStringList.Create;
  try
    SetCursor(HourGlassCursor);
    Status := getDriver(PChar(FVendorLib), PChar(AnsiString(Trim(FParams.Values[ERROR_RESOURCE_KEY]))), FSQLDriver);
    if Status <> DBXERR_NONE then
      DataBaseErrorFmt(sDLLLoadError, [FVendorLib]);

    SetLength(DrvVersionStr, 128);
    SQLConnection := TISQLConnection25;
    if FSQLDriver.GetOption(eDrvVersion, PChar(DrvVersionStr), Length(DrvVersionStr), PropSize) = DBXERR_NONE then
    begin
      SetLength(DrvVersionStr, PropSize);
      for ind := 1 to MaxDBXDrvTableEntry do
        if DBXDrvMap[ind].DrvVersion = DrvVersionStr then
        begin
          SQLConnection := DBXDrvMap[ind].SQLConnection;
          if FSQLDriver.SetOption(eDrvProductVersion, Integer(PChar(DBXDrvMap[ind].ProdVersion))) = DBXERR_NONE then
            break;
        end;
    end;

    Check(FSQLDriver.setOption(eDrvRestrict, GDAL));
    Check(FSQLDriver.getSQLConnection(NewSQLConnection));

    FISQLConnection := SQLConnection.Create(NewSQLConnection);

    for ind := 0 to FMonitorUsers.Count -1 do
      TSQLMonitor(FMonitorUsers[ind]).UpdateTraceCallBack;

    GetLoginParams(LoginParams);
    SetCursor(HourGlassCursor);
    RegisterTraceCallback(True);
    SetConnectionParams;
    FLoginUsername := LoginParams.Values[ szUSERNAME ];
    ConnectionStr := Params.Values[ CONNECTION_STRING ];
    if ConnectionStr <> '' then
      Check(Connection.connect)
    else
     Check(Connection.connect(PWideChar(trim(LoginParams.Values[DATABASENAME_KEY])), PWideChar(LoginParams.Values[ szUSERNAME ]),
         PWideChar(LoginParams.Values[ szPASSWORD ])));
    FISQLConnection.getOption(eConnMaxActiveComm, @FMaxStmtsPerConn, Sizeof(Integer), PropSize);

    Check(Connection.getSQLMetaData(FSQLMetaData));
    SetLength(str, 128);
    Status := FSQLMetaData.GetStringOption(eMetaDefaultSchemaName, str);
    if (Status = DBXERR_NONE) then
    begin
      FDefaultSchemaName := str;
      TrimmedUserName := str;
    end
    else
      TrimmedUserName := Trim(LoginParams.Values[ szUSERNAME ]);
    if TrimmedUserName <> '' then
      FSQLMetaData.SetStringOption(eMetaSchemaName, TrimmedUserName);
    ConnectionOptions;
    ConnectionState := csStateOpen;
  finally
    SetCursor(DefaultCursor);
    LoginParams.Free;
    if ConnectionState = csStateConnecting then // an exception occurred
    begin
      ConnectionState := csStateClosed;
      SQLDllHandle := THandle(0);
      if Assigned(FISQLConnection) then
        FreeAndNil(FISQLConnection)
    end;
  end;
end;

function TSQLConnection.GetLoginUsername : WideString;
begin
  Result := FLoginUserName;
end;


procedure TSQLConnection.GetLoginParams(LoginParams: TWideStrings);
var
  I: Integer;
  PName: string;
begin
  LoginParams.BeginUpdate;
  try
    LoginParams.Clear;
    for I := 0 to FParams.Count - 1 do
      begin
        if LoginParams.IndexOf(FParams[I]) > -1 then continue;
        PNAME := FParams.Names[I];
        if CompareText(PName, szPASSWORD) = 0 then
           LoginParams.Add(Wideformat('%s=%s',[szPASSWORD, FParams.Values[szPASSWORD] ]))
        else if CompareText(PName, szUSERNAME) = 0 then
           LoginParams.Add(Wideformat('%s=%s',[szUSERNAME, FParams.Values[szUSERNAME]]))
        else if CompareText(PName, DATABASENAME_KEY) = 0 then
          LoginParams.Add(Wideformat('%s=%s',[DATABASENAME_KEY, trim(FParams.Values[DATABASENAME_KEY])]));
      end;
  finally
    LoginParams.EndUpdate;
  end;
  if LoginPrompt then
     Login(LoginParams);
end;

function TSQLConnection.GetConnected: Boolean;
begin
  Result := Assigned(FISQLConnection) and (not
      (ConnectionState in [csStateClosed, csStateConnecting,
      csStateDisconnecting]));
end;

procedure TSQLConnection.DoDisconnect;
begin
  if FSQLDriver <> nil then
  begin
    ConnectionState := csStateDisconnecting;
    CloseDataSets;
    RegisterTraceCallback(False);
    if (FSQLMetaData <> nil) then
      FreeAndNil(FSQLMetaData);
    if (FISQLConnection <> nil) then
    begin
      FISQLConnection.disconnect;
      FTransactionCount := 0;
      FreeAndNil(FISQLConnection)
    end;
    SQLDllHandle := THandle(0);
    ConnectionState := csStateClosed;
    FSQLDriver := nil;
    FSelectStatements := 0;
    FPrevSelectStatements := 0;
  end;
  FParamsLoaded := False;
end;

procedure TSQLConnection.CloseDataSets;
var
  I: Integer;
begin
  for I := 0 to DataSetCount -1 do
  begin
    if TCustomSQLDataSet(DataSets[i]).Active then
      TCustomSQLDataSet(DataSets[i]).Close;
    TCustomSQLDataSet(DataSets[i]).FreeStatement;
  end;
  for I := 0 to FMonitorUsers.Count -1 do
  begin
    if Self.FIsCloned then
      TSQLMonitor(FMonitorUsers[I]).SwitchConnection( Self.FCloneParent );
  end;
end;

procedure TSQLConnection.CheckDisconnect;
var
  I: Integer;
begin
  if Connected and not (KeepConnection or InTransaction or (csLoading in ComponentState)) then
  begin
    for I := 0 to DataSetCount - 1 do
      if (DataSets[I].State <> dsInActive) then Exit;
    Close;
  end;
end;

procedure TSQLConnection.CheckInactive;
begin
  if FISQLConnection <> nil then
    if csDesigning in ComponentState then
      Close
    else
      DatabaseError(SdatabaseOpen, Self);
end;

procedure TSQLConnection.CheckActive;
begin
  if FISQLConnection = nil then DatabaseError(SDatabaseClosed, Self);
end;

{ Query execution }

function TSQLConnection.GetConnectionForStatement: TSQLConnection;
begin
  if (FMaxStmtsPerConn > 0) and (FSelectStatements >= FMaxStmtsPerConn)
       and (FSelectStatements > FPrevSelectStatements) and (FSelectStatements > 0)
       and not (FTransactionCount > 0) and AutoClone then
    Result := CloneConnection
  else
    Result := Self;
    FPrevSelectStatements := FSelectStatements;
end;

function TSQLConnection.ExecuteDirect(const SQL: WideString): Integer;
var
  Command: TISQLCommand;
  Cursor: TISQLCursor;
  Status: SQLResult;
  Connection: TSQLConnection;
  RowsetSize: Integer;
  CurSection : TSqlToken;
  PCommand: pWideChar;
  Value: WideString;
begin
  CheckConnection(eConnect);
  Cursor := nil;
  Result := 0;
  RowsetSize := defaultRowsetSize;
  PCommand := PWideChar(SQL);
  CurSection := stUnknown;
  CurSection := NextSQLToken(PCommand, Value, CurSection);
  if CurSection = stSelect then
    Inc(FSelectStatements);
  Connection := GetConnectionForStatement;
  if Connection.FISQLConnection.getSQLCommand(Command) = DBXERR_NONE then
  try

    if Params.Values[ROWSETSIZE_KEY] <> '' then
    try
      RowsetSize := StrToInt(trim(Params.Values[ROWSETSIZE_KEY]));
    except
      RowsetSize := defaultRowsetSize;
    end;
    Command.setOption(eCommRowsetSize, RowsetSize);

    Status := Command.executeImmediate(PWideChar(SQL), Cursor);

    if Status = DBXERR_NONE then
    begin
      try
        Status := Command.getRowsAffected(LongWord(Result));
        if not Assigned(Cursor) then
          Command.Close;
      finally
        FreeAndNil(Cursor);
      end;
    end;
    if Status <> DBXERR_NONE then
      SQLError(Status, exceptCommand, Command);
  finally
    FreeAndNil(Command);
  end;
end;

function TSQLConnection.Execute(const SQL: WideString; Params: TParams;
  ResultSet: Pointer = nil): Integer;
var
  Status: SQLResult;
  SQLText: WideString;
  RowsAffected: LongWord;
  DS: TCustomSQLDataSet;
  I, ParamCount: Integer;
begin
  Result := 0;
  DS := TCustomSQLDataSet.Create(nil);
  try
    CheckConnection(eConnect);
    SetCursor(HourGlassCursor);
    DS.SQLConnection := Self;
    ConnectionState := csStateExecuting;
    if (Params <> nil) and (Params.Count > 0) then
    begin
      SQLText := FixParams(SQL, Params.Count, Self.GetQuoteChar);
      ParamCount := Params.Count;
    end else
    begin
      SQLText := Copy(SQL, 1, Length(SQL));
      ParamCount := 0;
    end;
    DS.FCommandText := SQLText;
    if ResultSet = nil then
    begin
      DS.CheckStatement;
      Status := DS.FSQLCommand.prepare(PWideChar(SQLText), ParamCount);
      if Status <> DBXERR_NONE then
        SQLError(Status, exceptCommand, DS.FSQLCommand);
      if ParamCount > 0 then
        SetQueryProcParams(Self, DS.FSQLCommand, Params);
      Status := DS.FSQLCommand.execute(DS.FSQLCursor);
      if Status <> DBXERR_NONE then
        SQLError(Status, exceptCommand, DS.FSQLCommand);
      Status := DS.FSQLCommand.getRowsAffected(RowsAffected);
      if Status <> DBXERR_NONE then
        SQLError(Status, exceptCommand, DS.FSQLCommand);
      Result := RowsAffected;
    end
    else
    begin
      if ParamCount > 0 then
      begin
        for I := 0 to ParamCount -1 do
        begin
          DS.Params.CreateParam(Params.Items[I].DataType, format('P%d',[I+1]), ptInput);
          DS.Params[I].Value := Params[I].Value;
        end;
      end;
      DS.MaxBlobSize := DefaultMaxBlobSize;
      DS.Active := True;
    end;
  finally
    SetCursor(DefaultCursor);
    if ResultSet = nil then
      DS.Free
    else
      TCustomSQLDataSet(ResultSet^) := DS;
    ConnectionState := csStateOpen;
  end;
end;

{ Metadata retrieval }

function TSQLConnection.CloneConnection: TSQLConnection;
var
  SelfParent: TSQLConnection;
  I: Integer;
  Status: SQLResult;
  buf : WideString;
  Len : smallint;
begin      // do not allow nested clones
  if Self.FIsCloned then
    SelfParent := Self.FCloneParent
  else
    SelfParent := Self;
  Result := TSQLConnection.Create(nil);
  Result.FIsCloned := True;
  Result.FLoadParamsOnConnect := False;
  Result.LoginPrompt := False;
  Result.FDriverName := SelfParent.FDriverName;
  Result.FConnectionName := SelfParent.FConnectionName;
  Result.Name := SelfParent.Name + 'Clone1';
  Result.FParams.AddStrings(SelfParent.FParams);
  Result.FGetDriverFunc := SelfParent.FGetDriverFunc;
  Result.FLibraryName := SelfParent.FLibraryName;
  Result.FVendorLib := SelfParent.VendorLib;

  Len := 0;
  Status := FISQLConnection.getOption(eConnConnectionString, nil, 0, Len); // Len is number of byte.
  if (Status <> 0) or (Len <= 0) then
    Len := 1024;
  SetLength(buf, Len div sizeof(WideChar));
  FillChar(buf[1], Len div sizeof(WideChar), #0);
  Status := FISQLConnection.getStringOption(eConnConnectionString, buf);
  if Status = 0 then
    Result.Params.Values[CONNECTION_STRING] := PWideChar(buf);

  Result.FTableScope := SelfParent.TableScope;
  Result.Connected := Self.Connected;
  Result.FCloneParent := SelfParent;
  for I := 0 to FMonitorUsers.Count -1 do
    TSQLMonitor(FMonitorUsers[I]).SwitchConnection( Result );
end;

function TSQLConnection.OpenSchemaTable(eKind: TSchemaType; SInfo: WideString; SQualifier: WideString = ''; SPackage: WideString = ''): TCustomSQLDataSet;
begin
  Result := OpenSchemaTable(eKind, SInfo, SQualifier, SPackage , '');
end;

function TSQLConnection.OpenSchemaTable(eKind: TSchemaType; SInfo: WideString; SQualifier: WideString = ''; SPackage: WideString = ''; SSchemaName: WideString = ''): TCustomSQLDataSet;
var
  DataSet: TCustomSQLDataSet;
begin
  CheckConnection(eConnect);
  DataSet := TCustomSQLDataSet.Create(nil);
  try
    DataSet.SetConnection(Self);
    DataSet.SetSchemaInfo(eKind, SInfo, SQualifier, SPackage);
    DataSet.SchemaName := SSchemaName;
    DataSet.Active := True;
  except
    FreeSchemaTable(DataSet);
    DataSet := nil;
  end;
  Result := DataSet;
end;

procedure TSQLConnection.FreeSchemaTable(DataSet: TCustomSQLDataSet);
var
  SaveKeepConnection: Boolean;
begin
  if Assigned(Dataset) then
    FreeAndNil(DataSet.FClonedConnection);
  SaveKeepConnection := FKeepConnection;
  FKeepConnection := True;
  if Assigned(Dataset) then
    DataSet.Free;
  FKeepConnection := SaveKeepConnection;
end;

procedure TSQLConnection.OpenSchema(eKind: TSchemaType; sInfo: WideString; List: TWideStrings);
begin
  OpenSchema(eKind, sInfo, '', List);
end;

const
  TBL_NAME_FIELD = 'TABLE_NAME';           { Do not localize }
  PROC_NAME_FIELD = 'PROC_NAME';           { Do not localize }
  COL_NAME_FIELD = 'COLUMN_NAME';          { Do not localize }
  IDX_NAME_FIELD = 'INDEX_NAME';           { Do not localize }
  OBJECT_NAME_FIELD = 'OBJECT_NAME';      { Do not localize }

procedure TSQLConnection.OpenSchema(eKind: TSchemaType; sInfo, SSchemaName: WideString; List: TWideStrings);
var
  DataSet: TCustomSQLDataSet;
  NameField: TField;
  PackageName : WideString;
  ISList: TWideStringList;

begin
  CheckConnection(eConnect);
  if FISQLConnection = nil then
    DatabaseError(sConnectionNameMissing);
  DataSet := nil;
  NameField := nil;
  if eKind = stProcedures then
    PackageName := sInfo;
  CheckActive;
  SetCursor(HourGlassCursor);
  try
    DataSet := OpenSchemaTable(eKind, sInfo, '', PackageName, SSchemaName);
    if Assigned(DataSet) then
    begin
      case eKind of
        stColumns:
          NameField := DataSet.FieldByName(COL_NAME_FIELD);
        stProcedures:
          begin
            if not Assigned(DataSet) then DatabaseErrorFmt(SStoredProcsNotSupported, [FDriverName]);
            NameField := DataSet.FieldByName(PROC_NAME_FIELD);
          end;
        stPackages:
          begin
            if not Assigned(DataSet) then DatabaseErrorFmt(SPackagesNotSupported, [FDriverName]);
            NameField := DataSet.FieldByName(OBJECT_NAME_FIELD);
          end;
        stIndexes:
          NameField := DataSet.FieldByName(IDX_NAME_FIELD);
        stTables, stSysTables:
          NameField := DataSet.FieldByName(TBL_NAME_FIELD);
        stUserNames:
          NameField := DataSet.FieldByName(OBJECT_NAME_FIELD);
      end;
      if (not DataSet.Eof) then
      begin
        ISList:= TWideStringList.Create;
        try
          try
            ISList.BeginUpdate;
            ISList.Duplicates := dupIgnore;
            ISList.CaseSensitive := False;
            while not DataSet.Eof do
            begin
              ISList.Add(NameField.AsWideString);
              DataSet.Next;
            end;
            ISList.Sorted := True;
          finally
            ISList.EndUpdate;
          end;
          try
            List.BeginUpdate;
            List.Clear;
            List.AddStrings(ISList);
          finally
            List.EndUpdate;
          end;
        finally
          ISList.Free;
        end;
      end;
    end;
  finally
    SetCursor(DefaultCursor);
    if Assigned(DataSet) then FreeSchemaTable(DataSet);
  end;
end;

procedure TSQLConnection.GetFieldNames(const TableName: String; List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stColumns, TableName, wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetFieldNames(const TableName: string; SchemaName: String; List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stColumns, TableName, SchemaName, wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetFieldNames(const TableName: WideString; List: TWideStrings);
begin
  OpenSchema(stColumns, TableName, List);
end;

procedure TSQLConnection.GetFieldNames(const TableName: Widestring; SchemaName: WideString; List: TWideStrings);
begin
  OpenSchema(stColumns, TableName, SchemaName, List);
end;

procedure TSQLConnection.GetProcedureNames(List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stProcedures, '', wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetProcedureNames(const PackageName : string; List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stProcedures, PackageName, '',  wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetProcedureNames(const PackageName, SchemaName : string; List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stProcedures, PackageName, SchemaName, wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetProcedureNames(List: TWideStrings);
begin
  OpenSchema(stProcedures, '', List);
end;

procedure TSQLConnection.GetProcedureNames(const PackageName : WideString; List: TWideStrings);
begin
  OpenSchema(stProcedures, PackageName, '',  List);
end;

procedure TSQLConnection.GetProcedureNames(const PackageName, SchemaName : WideString; List: TWideStrings);
begin
  OpenSchema(stProcedures, PackageName, SchemaName, List);
end;

procedure TSQLConnection.GetPackageNames(List: TStrings);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stPackages, '', wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetPackageNames(List: TWideStrings);

begin
  OpenSchema(stPackages, '', List);
end;

procedure TSQLConnection.GetTableNames(List: TStrings; SystemTables: Boolean = False);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    GetTableNames( wList, '', SystemTables );
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetTableNames(List: TStrings; SchemaName: WideString; SystemTables: Boolean = False);
var
  wList: TWideStringList;
begin
  wList := TWideStringList.Create;
  try
    GetTableNames(wList, SchemaName, SystemTables);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetTableNames(List: TWideStrings; SystemTables: Boolean = False);
begin
  GetTableNames( List, '', SystemTables );
end;

procedure TSQLConnection.GetTableNames(List: TWideStrings; SchemaName: WideString; SystemTables: Boolean = False);
var
  eType: TSchemaType;
begin
  if SystemTables then
    eType := stSysTables
  else
    eType := stTables;
  OpenSchema(eType, '', SchemaName, List);
end;

procedure TSQLConnection.GetIndexNames(const TableName: string; List: TStrings);
var
  wList: TWideStrings;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stIndexes, TableName, '', wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetIndexNames(const TableName, SchemaName: string; List: TStrings);
var
  wList: TWideStrings;
begin
  wList := TWideStringList.Create;
  try
    OpenSchema(stIndexes, TableName, SchemaName, wList);
    List.Assign(wList);
  finally
    wList.Free;
  end;
end;

procedure TSQLConnection.GetIndexNames(const TableName: WideString; List: TWideStrings);
begin
  OpenSchema(stIndexes, TableName, '', List);
end;

procedure TSQLConnection.GetIndexNames(const TableName, SchemaName: WideString; List: TWideStrings);
begin
  OpenSchema(stIndexes, TableName, SchemaName, List);
end;

procedure TSQLConnection.GetProcedureParams(ProcedureName: WideString; List: TList);
begin
  GetProcedureParams(ProcedureName, '', List);
end;

procedure TSQLConnection.GetProcedureParams(ProcedureName, PackageName: WideString; List: TList);
begin
  GetProcedureParams(ProcedureName, PackageName, '',  List);
end;

procedure TSQLConnection.GetProcedureParams(ProcedureName, PackageName, SchemaName: WideString; List: TList);
const
  TypeFieldName = 'PARAM_TYPE';             { do not localize }
  DataTypeFieldName = 'PARAM_DATATYPE';     { do not localize }
  SubTypeFieldName = 'PARAM_SUBTYPE';       { do not localize }
  PosFieldName = 'PARAM_POSITION';          { do not localize }
  PrecisionFieldName = 'PARAM_PRECISION';   { do not localize }
  ScaleFieldName = 'PARAM_SCALE';           { do not localize }
  LengthFieldName = 'PARAM_LENGTH';         { do not localize }
  ParamNameFieldName = 'PARAM_NAME';        { do not localize }
  ResultParam = 'Result';                   { Do not localize }
var
  DataSet: TCustomSQLDataSet;
  ArgDesc: pSPParamDesc;
  V: Variant;
begin
  DataSet := nil;
  try
    DataSet := OpenSchemaTable(stProcedureParams, ProcedureName,'', PackageName, SchemaName);
    if not Assigned(DataSet) then SQLError(SQLResult(-1), exceptMetadata);
    while not DataSet.EOF do
    begin
      New(ArgDesc);
      ArgDesc^.iParamNum := DataSet.FieldByName(PosFieldName).Value;
      V := DataSet.FieldByName(TypeFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.iArgType := ptUnknown
      else
        ArgDesc^.iArgType := V;
      V := DataSet.FieldByName(DataTypeFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.iDataType := ftUnknown
      else
        ArgDesc^.iDataType := DataTypeMap[Integer(V)];
      V := DataSet.FieldByName(SubTypeFieldName).Value;
      if not VarIsNull(V) then
        if V = fldstFIXED then
          ArgDesc^.iDataType := ftFixedChar;
      V := DataSet.FieldByName(PrecisionFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.iUnits1 := 0
      else
        ArgDesc^.iUnits1 := V;
      V := DataSet.FieldByName(ScaleFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.iUnits2 := 0
      else
        ArgDesc^.iUnits2 := V;
      V := DataSet.FieldByName(LengthFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.iLen := 0
      else
        ArgDesc^.iLen := V;
      V := DataSet.FieldByName(ParamNameFieldName).Value;
      if VarIsNull(V) then
        ArgDesc^.szName := ResultParam
      else
        ArgDesc^.szName := V;
      List.Add(ArgDesc);
      DataSet.next;
    end;
  finally
    FreeSchemaTable(DataSet);
  end;
end;

{ trace }

procedure TSQLConnection.SetTraceCallbackEvent(Event: TSQLCallbackEvent; IClientInfo: Integer);
begin
  FTraceCallbackEvent := Event;
  FTraceClientData := IClientInfo;
  if Connected and not (csLoading in ComponentState) then
    RegisterTraceCallBack(Assigned(Event) and (IClientInfo > 0));
end;

procedure TSQLConnection.RegisterTraceCallback(Value: Boolean);
begin
  if (Value) then
  begin
    if Assigned(FTraceCallbackEvent) and (FTraceClientData <> 0) then
    begin
      Check(FISQLConnection.SetOption(
           TSQLConnectionOption(eConnCallBack), Integer(@FTraceCallbackEvent)));
      Check(FISQLConnection.SetOption(
           TSQLConnectionOption(eConnCallBackInfo), Integer(FTraceClientData)));
    end;
  end else
  begin
    if Assigned(FISQLConnection) then
    begin
      Check(FISQLConnection.SetOption(
            TSQLConnectionOption(eConnCallback), Integer(0)));
      Check(FISQLConnection.SetOption(
            TSQLConnectionOption(eConnCallBackInfo), Integer(0)));
    end;
  end;
end;

{ transaction support }

function TSQLConnection.GetInTransaction: Boolean;
begin
  Result := FTransactionCount > 0;
end;

procedure TSQLConnection.StartTransaction( TransDesc: TTransactionDesc);
var
  Status: SQLResult;
begin
  CheckConnection(eConnect);
  if Connected then
  begin
    if FTransactionsSupported then
    begin
      CheckActive;
      if (not InTransaction) or FSupportsMultiTrans then
      begin
        Status := FISQLConnection.beginTransaction(LongWord(@TransDesc));
        if not ( Status in [DBXERR_NONE, DBXERR_NOTSUPPORTED] ) then
          Check(Status);
        Inc(FTransactionCount);
      end else
        DatabaseError(sActiveTrans, self)
    end;
  end else
    DatabaseError(SDatabaseClosed, Self);
end;

procedure TSQLConnection.Rollback( TransDesc: TTransactionDesc);
var
  Status: SQLResult;
begin
  if FTransactionsSupported then
  begin
    if InTransaction then
    begin
      if Assigned(FISQLConnection) then
      begin
        Status := FISQLConnection.rollback(LongWord(@TransDesc));
        if not ( Status in [DBXERR_NONE, DBXERR_NOTSUPPORTED] ) then
          Check(Status);
        Dec(FTransactionCount);
      end
      else
        DatabaseError(SDatabaseClosed, Self);
    end else
      DatabaseError(sNoActiveTrans, self);
    CheckDisconnect;
  end;
end;

procedure TSQLConnection.Commit(TransDesc: TTransactionDesc);
var
  Status: SQLResult;
begin
  if FTransactionsSupported then
  begin
    if InTransaction then
    begin
      if Assigned(FISQLConnection) then
      begin
        Status := FISQLConnection.Commit(LongWord(@TransDesc));
        if not ( Status in [DBXERR_NONE, DBXERR_NOTSUPPORTED] ) then
          Check(Status);
        Dec(FTransactionCount);
      end
      else
        DatabaseError(SDatabaseClosed, Self);
    end
    else
      DatabaseError(sNoActiveTrans, self);
    CheckDisconnect;
  end;
end;

function TSQLConnection.GetDataSet(Index: Integer): TCustomSQLDataSet;
begin
  Result := TCustomSQLDataSet(inherited GetDataSet(Index));
end;

{ misc. property set/get }

procedure TSQLConnection.SetDriverName(Value: string);

  procedure LoadDriverParams;
  var
    Index: Integer;
  begin
    FConnectionName := DriverName;
    LoadParamsFromIniFile(DriverRegistryFile);
    FConnectionName := '';
    Index := Params.IndexOfName(VENDORLIB_KEY);
    if Index <> -1 then
      Params.Delete(Index);
    Index := Params.IndexOfName(DLLLIB_KEY);
    if Index <> -1 then
      Params.Delete(Index);
    Index := Params.IndexOfName(GETDRIVERFUNC_KEY);
    if Index <> -1 then
      Params.Delete(Index);
  end;

begin
  if FDriverName <> Value then
  begin
    CheckInactive;
    if FConnectionName = '' then
    begin
      FVendorLib := '';
      FLibraryName := '';
      FGetDriverFunc := '';
      FParams.Clear;
    end;
    FDriverName := Value;
    if (csDesigning in ComponentState) and not (csLoading in ComponentState) then
    begin
      FParams.Clear;
      FParamsLoaded := False;
      if FDriverName <> '' then
      begin
        try
          FVendorLib := Trim(GetProfileString(FDriverName, VENDORLIB_KEY, DriverRegistryFile));
          FLibraryName := Trim(GetProfileString(FDriverName, DLLLIB_KEY, DriverRegistryFile));
          FGetDriverFunc := Trim(GetProfileString(FDriverName, GETDRIVERFUNC_KEY, DriverRegistryFile));
          if FConnectionName = '' then
            LoadDriverParams;
        except
          DatabaseErrorFmt(SDriverNotInConfigFile, [Value, DriverRegistryFile]);
        end;
      end;
    end;
  end;
end;

function TSQLConnection.GetFDriverRegistryFile: string;
begin
  if FDriverRegistryFile = '' then
    FDriverRegistryFile := GetDriverRegistryFile(csDesigning in ComponentState);
  Result := FDriverRegistryFile;
end;

function TSQLConnection.GetConnectionName: string;
begin
  Result := FConnectionName;
end;

procedure TSQLConnection.SetConnectionName(Value: string);
var
  NewDriver: string;
begin
  if FConnectionName <> Value then
  begin
    FLastError := '';
    if not (csLoading in ComponentState) then
      if Connected then Connected := False;
    if (FDriverName = '') and (Value = '') then
    begin
      FVendorLib := '';
      FLibraryName := '';
      FParams.Clear;
    end;
    FParamsLoaded := False;
    FConnectionName := Value;
    if not (csLoading in ComponentState) then
      CloseDataSets;
    if (csDesigning in ComponentState) and not (csLoading in ComponentState) then
    begin
      if (Value = '') and (LoadParamsOnConnect) then
        FParams.Clear;
      if Value <> '' then
      begin
        NewDriver := GetProfileString(FConnectionName, DRIVERNAME_KEY, ConnectionRegistryFile);
        if NewDriver <> DriverName then
          DriverName := NewDriver;
        LoadParamsFromIniFile;
      end;
    end;
  end;
end;

function TSQLConnection.GetVendorLib: string;
begin
  Result := FVendorLib;
  if (Result = '') and (FLoadParamsOnConnect or (csLoading in ComponentState)) then
    Result := Trim(GetProfileString(FDriverName, VENDORLIB_KEY, DriverRegistryFile));
end;

function TSQLConnection.GetLibraryName: string;
begin
  Result := FLibraryName;
  if (Result = '') and (FLoadParamsOnConnect or (csLoading in ComponentState)) then
    Result := Trim(GetProfileString(FDriverName, DLLLIB_KEY, DriverRegistryFile));
end;

procedure TSQLConnection.SetConnectionParams;
var
  ServerCharSet, STransIsolationKey, ConnectionStr: WideString;
  ILevel: TTransIsolationLevel;
begin

  if FParams.Values[HOSTNAME_KEY] <> '' then
    FISQLConnection.SetStringOption(eConnHostName, trim(FParams.Values[HOSTNAME_KEY]));
  if FParams.Values[ROLENAME_KEY] <> '' then
    FISQLConnection.SetStringOption(eConnRoleName, trim(FParams.Values[ROLENAME_KEY]));
  if FParams.Values[WAITONLOCKS_KEY] <> '' then
    FISQLConnection.SetOption(eConnWaitOnLocks, LongInt(WideUpperCase(trim(FParams.Values[WAITONLOCKS_KEY])) = 'TRUE'));
  if FParams.Values[COMMITRETAIN_KEY] <> '' then
    FISQLConnection.SetOption(eConnCommitRetain, LongInt(WideUpperCase(trim(FParams.Values[COMMITRETAIN_KEY])) = 'TRUE'));
  if FParams.Values[AUTOCOMMIT_KEY] <> '' then
    FISQLConnection.SetOption(eConnAutoCommit, LongInt(WideUpperCase(trim(FParams.Values[AUTOCOMMIT_KEY])) = 'TRUE'));
  if FParams.Values[BLOCKINGMODE_KEY] <> '' then
    FISQLConnection.SetOption(eConnBlockingMode, LongInt(WideUpperCase(trim(FParams.Values[BLOCKINGMODE_KEY])) = 'TRUE'));
  ServerCharSet := trim(FParams.Values[SQLSERVER_CHARSET_KEY]);
  if ServerCharSet <> '' then
    FISQLConnection.SetOption(eConnServerCharSet, LongInt(PWideChar(ServerCharSet)));
  ConnectionStr := trim(FParams.Values[CONNECTION_STRING]);
  if ConnectionStr <> '' then
    FISQLConnection.SetStringOption(eConnConnectionString, ConnectionStr);

  FTransIsoLevel := xilReadCommitted;
  STransIsolationKey := Format(TRANSISOLATION_KEY, [DriverName]);
  if FParams.Values[STransIsolationKey] <> '' then
  begin
    if WideLowerCase(FParams.Values[STransIsolationKey]) = SRepeatRead then
      ILevel := xilRepeatableRead
    else if WideLowerCase(FParams.Values[STransIsolationKey]) = SDirtyRead then
      ILevel := xilDirtyRead
    else
      ILevel := xilReadCommitted;
    FTransIsoLevel := ILevel;
    FISQLConnection.SetOption(eConnTxnIsoLevel, LongInt(ILevel));
  end;
  if FParams.Values[SQLDIALECT_KEY] <> '' then
    FISQLConnection.SetOption(eConnSQLDialect, LongInt(StrToInt(trim(FParams.Values[SQLDIALECT_KEY]))));

  if FParams.Values[OSAUTHENTICATION] <> '' then
    FISQLConnection.SetOption(eConnOSAuthentication, LongInt(WideUpperCase(trim(FParams.Values[OSAUTHENTICATION])) = 'TRUE'));
  if FParams.Values[COMPRESSED] <> '' then
    FISQLConnection.SetOption(eConnCompressed, LongInt(WideUpperCase(trim(FParams.Values[COMPRESSED])) = 'TRUE'));
  if FParams.Values[ENCRYPTED] <> '' then
    FISQLConnection.SetOption(eConnCompressed, LongInt(WideUpperCase(trim(FParams.Values[ENCRYPTED])) = 'TRUE'));
  if FParams.Values[SERVERPORT] <> '' then
    FISQLConnection.SetOption(eConnServerPort, LongInt(trim(FParams.Values[SERVERPORT])));
  if FParams.Values[MULTITRANSENABLED] <> '' then
    FISQLConnection.SetOption(eConnMultipleTransaction, LongInt(WideUpperCase(trim(FParams.Values[MULTITRANSENABLED])) = 'TRUE'));
  if FParams.Values[TRIMCHAR] <> '' then
    FISQLConnection.SetOption(eConnTrimChar, LongInt(WideUpperCase(trim(FParams.Values[TRIMCHAR])) = 'TRUE'));

  if FParams.Values[CUSTOM_INFO] <> '' then
    FISQLConnection.SetOption(eConnCustomInfo, LongInt(trim(FParams.Values[CUSTOM_INFO])));
  if FParams.Values[CONN_TIMEOUT] <> '' then
    FISQLConnection.SetOption(eConnTimeOut, LongInt(StrToInt(trim(FParams.Values[CONN_TIMEOUT]))));

  if FParams.Values[TDSPACKETSIZE] <> '' then
    FISQLConnection.SetOption(eConnTDSPacketSize, LongInt(StrToInt(trim(FParams.Values[TDSPACKETSIZE]))));
  if FParams.Values[CLIENTHOSTNAME] <> '' then
    FISQLConnection.SetStringOption(eConnClientHostName, trim(FParams.Values[CLIENTHOSTNAME]));
  if FParams.Values[CLIENTAPPNAME] <> '' then
    FISQLConnection.SetStringOption(eConnClientAppName, trim(FParams.Values[CLIENTAPPNAME]));
  if FParams.Values[PREPARESQL] <> '' then
    FISQLConnection.SetOption(eConnPrepareSQL, LongInt(WideUpperCase(trim(FParams.Values[PREPARESQL])) = 'TRUE'));
  if FParams.Values[DECIMALSEPARATOR] <> '' then
    FISQLConnection.SetOption(eConnDecimalSeparator, LongInt(trim(FParams.Values[DECIMALSEPARATOR])));

end;

procedure TSQLConnection.LoadParamsFromIniFile(FFileName: WideString = '');
var
  IniFile: TMemIniFile;
  List: TStrings;
  FIniFileName: string;
begin
  if not FParamsLoaded then
  begin
    if FConnectionName = '' then
      DatabaseError(SConnectionNameMissing);
    List := TStringList.Create;
    try
      if FFileName = '' then
        FIniFileName := ConnectionRegistryFile
      else
        FIniFileName := FFileName;
      IniFile := TMemIniFile.Create(FIniFileName);
      try
        if FileExists(FIniFileName) then
        begin
          IniFile.ReadSectionValues(FConnectionName, List);
          Params.BeginUpdate;
          try
            Params.Clear;
            Params.AddStrings(List);
          finally
            Params.EndUpdate;
          end;
        end else
          DatabaseErrorFmt(sMissingDriverRegFile, [FIniFileName]);
      finally
        IniFile.Free;
      end;
    finally
      List.Free;
    end;
    FParamsLoaded := True;
  end;
end;

procedure TSQLConnection.SetLocaleCode(Value: TLocaleCode);
begin
  FParams.Values[SQLLOCALE_CODE_KEY] := IntToHex(Value, 4);
end;

function TSQLConnection.GetLocaleCode: TLocaleCode;
begin
  if FParams.Values[SQLLOCALE_CODE_KEY] <> '' then
    Result := StrToInt(HexDisplayPrefix + FParams.Values[SQLLOCALE_CODE_KEY])
  else
    Result := 0;
end;

procedure TSQLConnection.SetKeepConnection(Value: Boolean);
begin
  if FKeepConnection <> Value then
  begin
    FKeepConnection := Value;
    if not Value and (FRefCount = 0) then Close;
  end;
end;

procedure TSQLConnection.SetParams(Value: TWideStrings);
begin
  CheckInactive;
  FParams.Assign(Value);
end;

function TSQLConnection.Check(Status: SQLResult): SQLResult;
begin
  if Status <> 0 then SQLError(Status, exceptConnection);
  Result := Status;
end;

procedure TSQLConnection.Loaded;
begin
  inherited Loaded;
end;

procedure TSQLConnection.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
end;

procedure TSQLConnection.GetSchemaNames(List: TStrings);
var
  S : TWideStrings;
begin
  S := TWideStringList.Create;
  try
    OpenSchema(stUserNames, '', S);
    List.Assign(S);
  finally
    S.Free;
  end;
end;

procedure TSQLConnection.GetSchemaNames(List: TWideStrings);
begin
  OpenSchema(stUserNames, '', List);
end;

function TSQLConnection.GetDefaultSchemaName: WideString;
begin
  CheckConnection(eConnect);
  Result := FDefaultSchemaName;
end;

{ TSQLDataLink }

constructor TSQLDataLink.Create(ADataSet: TCustomSQLDataSet);
begin
  inherited Create;
  FSQLDataSet := ADataSet;
end;

procedure TSQLDataLink.ActiveChanged;
begin
  if FSQLDataSet.Active then FSQLDataSet.RefreshParams;
end;

function TSQLDataLink.GetDetailDataSet: TDataSet;
begin
  Result := FSQLDataSet;
end;

procedure TSQLDataLink.RecordChanged(Field: TField);
begin
  if (Field = nil) and FSQLDataSet.Active then FSQLDataSet.RefreshParams;
end;

procedure TSQLDataLink.CheckBrowseMode;
begin
  if FSQLDataSet.Active then FSQLDataSet.CheckBrowseMode;
end;

{ TCustomSQLDataSet }

constructor TCustomSQLDataSet.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FParams := TParams.Create(Self);
  FDataLink := TSQLDataLink.Create(Self);
  FIndexDefs := TIndexDefs.Create(Self);
  FCommandType := ctQuery;
  FCommandText := '';
  FParamCheck := True;
  FRecords := -1;
  FParamCount := -1;
  FSchemaInfo.FType := stNoSchema;
  SetUniDirectional(True);
  ObjectView := False;
end;

destructor TCustomSQLDataSet.Destroy;
begin
  Close;
  if Assigned(FSQLCursor) then FreeCursor;
  if Assigned(FSQLCommand) then FreeStatement;
  FreeAndNil(FParams);
  FreeAndNil(FIndexDefs);
  SetConnection(nil);
  FreeProcParams(FProcParams);
  inherited Destroy;
  FDataLink.Free;
  FreeBuffers;
end;

{ connection management }

procedure TCustomSQLDataSet.CheckConnection(eFlag: eConnectFlag);
begin
  if (FSQLConnection <> nil) then
    FSQLConnection.CheckConnection(eFlag)
  else if (eFlag in [eConnect, eReconnect ]) then
    DatabaseError(SMissingSQLConnection);
end;

procedure TCustomSQLDataSet.SetConnection(const Value: TSQLConnection);
begin
  CheckInactive;
  if Assigned(FSQLConnection) then
    FSQLConnection.UnRegisterClient(Self);
  FSQLConnection := Value;
  if (not (csLoading in ComponentState)) and (FSQLConnection <> Value) then
    SchemaName := '';
  if Assigned(FSQLConnection) then
  begin
    FSQLConnection.RegisterClient(Self,nil);
    if FMaxBlobSize = 0 then   // means it hasn't been changed
    begin
      if FSQLConnection.Params.Values[MAXBLOBSIZE_KEY] <> '' then
      try
        FMaxBlobSize := StrToInt(trim(FSQLConnection.Params.Values[MAXBLOBSIZE_KEY]));
      except
        FMaxBlobSize := DefaultMaxBlobSize;
      end else
        FMaxBlobSize := DefaultMaxBlobSize;
    end;
  end;
end;

function TCustomSQLDataSet.GetInternalConnection: TSQLConnection;
begin
  if Assigned(FClonedConnection) then
    Result := FClonedConnection
  else
    Result := FSQLConnection;
end;

{ Error Handling routine }
procedure TCustomSQLDataSet.SQLError(OpStatus: SQLResult; eType: TSQLExceptionType);
var
  dbxErrorMsg, ServerErrorMsg, ExceptionMessage: WideString;
  Message: WideString;
  Status: SQLResult;
begin
  dbxErrorMsg := '';
  ServerErrorMsg := '';
  ExceptionMessage := '';
  Status := SQL_NULL_DATA;
  if (OpStatus > 0) and (OpStatus <=  DBX_MAXSTATICERRORS) then
  begin
    if OpStatus = 64 then dbxErrorMsg := WideFormat(SDBXError, [SqlConst.SNODATA])
    else if OpStatus = 65 then dbxErrorMsg := WideFormat(SDBXError, [SqlConst.SSQLERROR])
    else dbxErrorMsg := WideFormat(SDBXError, [DbxError[OpStatus]]);
  end;
  case eType of
    exceptUseLast:
      Status := DBXERR_OUTOFRANGE;
    exceptCursor:
      Status := FSQLCursor.getErrorMessage(Message);
    exceptCommand:
      Status := FSQLCommand.getErrorMessage(Message);
  end;
  if Status = DBXERR_NONE then
    if Length(Message) > 0 then
      ServerErrorMsg := WideFormat(SSQLServerError, [Message]);
  if Length(dbxErrorMsg) > 0 then
    ExceptionMessage := dbxErrorMsg;
  if Length(ServerErrorMsg) > 0 then
  begin
    if Length(ExceptionMessage) > 0 then
      ExceptionMessage := ExceptionMessage + #13 + #10;
    ExceptionMessage := ExceptionMessage + ServerErrorMsg;
  end;
  if (Length(ExceptionMessage) = 0) and (LastError <> '') then
    ExceptionMessage := LastError;
  if Length(ExceptionMessage) = 0 then
    ExceptionMessage :=  WideFormat(SDBXUNKNOWNERROR, [intToStr(OpStatus)]);
  FLastError := ExceptionMessage;
  DatabaseError(ExceptionMessage);
end;

{ open/close Cursors and Statements }

procedure TCustomSQLDataSet.GetObjectTypeNames(Fields: TFields);
var
  I: Integer;
  ObjectField: TObjectField;
begin
  for I := 0 to Fields.Count - 1 do
  begin
    if Fields[I] is TObjectField then
    begin
      ObjectField := TObjectField(Fields[I]);
      ObjectField.ObjectType := FSQLCursor.getCurObjectTypeName(ObjectField.FieldNo);
      with ObjectField do
        if DataType in [ftADT, ftArray] then
        begin
          if (DataType = ftArray) and SparseArrays and
             (Fields[0].DataType = ftADT) then
            GetObjectTypeNames(TObjectField(Fields[0]).Fields) else
            GetObjectTypeNames(Fields);
        end;
    end;
  end;
end;

procedure TCustomSQLDataSet.InternalOpen;
begin
  ExecuteStatement;
  if not Assigned(FSQLCursor) then
  begin
    FSQLCommand.Close;
    FStatementOpen := False;
    if not FGetNextRecordSet then
      DataBaseError(SNoCursor,Self)
    else
      Exit;
  end;
  FieldDefs.Update;
  if DefaultFields then CreateFields;
  BindFields(True);
  if ObjectView then GetObjectTypeNames(Fields);
  InitBuffers;
end;

function TCustomSQLDataSet.IsCursorOpen: Boolean;
begin
  Result := (FSQLCursor <> nil);
end;

procedure TCustomSQLDataSet.OpenCursor(InfoQuery: Boolean);
begin
  if (SchemaInfo.FType = stNoSchema) and (FCommandText = '') then
    DatabaseError(SNoSQLStatement);
  CheckConnection(eConnect);
  SetPrepared(True);
  CheckPrepareError;
  if FDataLink.DataSource <> nil then
     SetParamsFromCursor;
  if (SchemaInfo.FType = stNoSchema) then
    Inc(FSqlConnection.FActiveStatements);
  inherited OpenCursor;
end;

procedure TCustomSQLDataSet.CloseCursor;
begin
  inherited CloseCursor;
  if (SchemaInfo.FType = stNoSchema) and (FSqlConnection <> nil) then
    Dec(FSqlConnection.FActiveStatements);
end;

procedure TCustomSQLDataSet.FreeCursor;
begin
  if Assigned(FSQLCursor) then
  begin
    FreeAndNil(FSQLCursor);
    FStatementOpen := False;   // Releasing Cursor closes associated statement
  end;
end;

procedure TCustomSQLDataSet.FreeStatement;
begin
  if Assigned(FSQLCommand) then
  begin
    FreeCursor;
    CloseStatement;
    FreeAndNil(FSQLCommand);
    if Assigned(FSQLConnection) then
      if Assigned(FClonedConnection) then
        FreeAndNil(FClonedConnection)
      else
       if FSQLConnection.FSelectStatements > 0 then
         Dec(FSQLConnection.FSelectStatements);
    FPrepared := False;
    FParamCount := -1;
  end
  else
  if (FSchemaInfo.FType <> stNoSchema) then
    if Assigned(FClonedConnection) then
      FreeAndNil(FClonedConnection)
    else
      if Assigned(FSQLConnection) and (FSQLConnection.FSelectStatements > 0) then
        Dec(FSQLConnection.FSelectStatements);

  if Assigned(FieldDefs) then
    FieldDefs.Updated := False;
  ClearIndexDefs;
end;

procedure TCustomSQLDataSet.CloseStatement;
begin
  FParamCount := -1;
  if Assigned(FSQLCommand) and FStatementOpen then
    FSQLCommand.Close;
end;

procedure TCustomSQLDataSet.InternalClose;
var
  DetailList: TList;
  I: Integer;
begin
  BindFields(False);
  if DefaultFields then DestroyFields;
  FreeBuffers;
  DetailList := TList.Create;
  try
    GetDetailDataSets(DetailList);
    for I := 0 to DetailList.Count -1 do
    if TDataSet(DetailList[I]) is TCustomSQLDataSet then
    begin
      TCustomSQLDataSet(TDataSet(DetailList[I])).Close;
      TCustomSQLDataSet(TDataSet(DetailList[I])).SetPrepared(False);
    end;
  finally
    DetailList.Free;
  end;
  if Assigned(FSQLConnection) and ((FSQLConnection.KeepConnection) or
     (FSQLConnection.DataSetCount > 1)) then
    FreeCursor
  else
    SetPrepared(False);
end;

procedure TCustomSQLDataSet.Loaded;
begin
  inherited Loaded;
end;

procedure TCustomSQLDataSet.InternalRefresh;
begin
  SetState(dsInactive);
  CloseCursor;
  OpenCursor(False);
  SetState(dsBrowse);
end;

procedure TCustomSQLDataSet.InitBuffers;
begin
  if (MaxBlobSize > 0) then
    SetLength(FBlobBuffer, MaxBlobSize * 1024);
  if (CalcFieldsSize > 0) then
    FCalcFieldsBuffer := AllocMem(CalcFieldsSize);
end;

procedure TCustomSQLDataSet.ClearIndexDefs;
begin
  FIndexDefs.Clear;
  FIndexDefsLoaded := False;
end;

procedure TCustomSQLDataSet.FreeBuffers;
begin
  if FBlobBuffer <> nil then
    SetLength(FBlobBuffer, 0);
  if FCalcFieldsBuffer <> nil then
  begin
    FreeMem(FCalcFieldsBuffer);
    FCalcFieldsBuffer := nil;
  end;
end;

procedure TCustomSQLDataSet.InitRecord(Buffer: PChar);
begin
  { NOP }
end;

procedure TCustomSQLDataSet.SetBufListSize(Value: Integer);
begin
end;

{ Cursor Level Metadata }

procedure TCustomSQLDataSet.AddFieldDesc(FieldDescs: TFieldDescList; DescNo: Integer;
  var FieldID: Integer; RequiredFields: TBits; FieldDefs: TFieldDefs);
const
  ArrayIndex = '[0]';
var
  FType: TFieldType;
  FSize: LongWord;
  FRequired: Boolean;
  FPrecision, I: Integer;
  FieldName, FName: WideString;
  FieldDesc: TFLDDesc;
  FldDef: TFieldDef;
begin
  FieldDesc := FieldDescs[DescNo];
  with FieldDesc do
  begin
    FieldName := szName;                                            
    FName := FieldName;
    I := 0;
    while FieldDefs.IndexOf(FName) >= 0 do
    begin
      Inc(I);
      FName := Format('%s_%d', [FieldName, I]);
    end;
    if iFldType < MAXLOGFLDTYPES then
      FType := DataTypeMap[iFldType]
    else
      FType := ftUnknown;
    if iFldType in [fldFMTBCD, fldBCD] then
    begin
      iUnits2 := Abs(iUnits2);
      if iUnits1 < iUnits2 then   // iUnits1 indicates Oracle 'usable decimals'
        iUnits1 := iUnits2;
      // ftBCD supports only up to 18-4.  If Prec > 14 or Scale > 4, make FMTBcd
      if (iUnits1 > (MaxBcdPrecision-4)) or (iUnits2 > MaxBcdScale) or FNumericMapping then
      begin
        FType := ftFMTBcd;
        iFldType := fldFMTBCD;
        if (iUnits1 = 38) and (iUnits2 in [0,38]) then
        begin
          iUnits1 := 32;
          iUnits2 := 8;
        end;
        if iUnits1 > MaxFMTBcdDigits then
          iUnits1 := MaxFMTBcdDigits;
      end;
    end;
    FSize := 0;
    FPrecision := 0;
    if RequiredFields.Size > FieldID then
      FRequired := RequiredFields[FieldID] else
      FRequired := False;
    case iFldType of
      fldZSTRING:
        begin
          if iUnits1 = 0 then { Ignore MLSLABEL field type on Oracle }
            FType := ftUnknown else
            FSize := iUnits1;
        end;
      fldWIDESTRING:
        begin
          if iUnits1 = 0 then { Ignore MLSLABEL field type on Oracle }
            FType := ftUnknown else
            FSize := iUnits1;
        end;

      fldBYTES, fldVARBYTES, fldRef:
        begin
          if iUnits1 = 0 then { Ignore MLSLABEL field type on Oracle }
            FType := ftUnknown else
            FSize := iUnits1;
        end;
      fldINT16, fldUINT16:
        if iLen <> 2 then FType := ftUnknown;
      fldINT32:
        if iSubType = fldstAUTOINC then
        begin
          FType := ftAutoInc;
          FRequired := False;
        end;
      fldFLOAT:
        if iSubType = fldstMONEY then FType := ftCurrency;
      fldFMTBCD, fldBCD:
        begin
          FSize := Abs(iUnits2);
          FPrecision := iUnits1;
        end;
      fldADT, fldARRAY:
        begin
          FSize := iUnits2;
          FPrecision := iUnits1;
        end;
      fldBLOB:
        begin
          FSize := iUnits1;
          if (iSubType >= fldstMEMO) and (iSubType <= fldstBFILE) then
            FType := BlobTypeMap[iSubType];
        end;
    end;
    FldDef := FieldDefs.AddFieldDef;
    with FldDef do
    begin
      FieldNo := FieldID;
      Inc(FieldID);
      Name := FName;
      DataType := FType;
      Size := FSize;
      Precision := FPrecision;
      if FRequired then
        Attributes := [faRequired];
      if efldrRights = fldrREADONLY then
        Attributes := Attributes + [faReadonly];
      if iSubType = fldstFIXED then
        Attributes := Attributes + [faFixed];
      InternalCalcField := bCalcField;
      case FType of
        ftADT:
          begin
            if iSubType = fldstADTNestedTable then
              Attributes := Attributes + [faUnNamed];
            for I := 1 to iUnits1 do
            begin
              LoadFieldDef(Word(FieldNo + I), FieldDescs[1]);
              AddFieldDesc(FieldDescs, 1, FieldID, RequiredFields, ChildDefs);
            end;
          end;
        ftArray:
          begin
            for I := 1 to iUnits1 do
            begin
              LoadFieldDef(Word(FieldNo + I), FieldDescs[1]);
              FieldDescs[1].szName := FieldDesc.szName + ArrayIndex;
              AddFieldDesc(FieldDescs, 1, FieldID, RequiredFields, ChildDefs);
            end;
          end;
      end;
    end;
  end;
end;

procedure TCustomSQLDataSet.LoadFieldDef(FieldID: Word; var FldDesc: TFLDDesc);
var
  iFldType: Word;
  iSubType: Word;
  iUnits1: SmallInt;
  iUnits2: SmallInt;
  iLen: LongWord;
  ReadOnly: LongBool;
begin
  FldDesc.iFldNum := FieldID;
  FldDesc.szName := FSQLCursor.getColumnName(FieldId);
  FSQLCursor.getColumnType(FieldId, iFldType, iSubtype);
  FldDesc.iFldType := iFldType;
  FldDesc.iSubtype := iSubtype;
  FSQLCursor.getColumnLength(FieldId, iLen);
  FldDesc.iLen := iLen;
  FSQLCursor.getColumnPrecision(FieldId, iUnits1);
  FldDesc.iUnits1 := iUnits1;
  FSQLCursor.getColumnScale(FieldId, iUnits2);
  FldDesc.iUnits2 := iUnits2;
  FSQLCursor.isReadOnly(FieldID, ReadOnly);
  if ReadOnly then
    FldDesc.efldrRights := fldrREADONLY;
end;

procedure TCustomSQLDataSet.InternalInitFieldDefs;
var
  FID: Integer;
  FieldDescs: TFieldDescList;
  RequiredFields: TBits;
  Nullable: LongBool;
  FldDescCount: Word;
begin
  if (FSQLCursor <> nil) then
  begin
    RequiredFields := TBits.Create;
    try
      FSQLCursor.getColumnCount(FldDescCount);
      SetLength(FieldDescs, FldDescCount);
      for FID := 1 to FldDescCount do
        FieldDescs[FID-1] := SQLConnection.Connection.getFldDescClass.Create;
      try
        RequiredFields.Size := FldDescCount + 1;
        FieldDefs.Clear;
        FID := 1;
        FMaxColSize := FldDescCount;
        while FID <= FldDescCount do
        begin
          FSQLCursor.IsNullable(Word(FID), Nullable);
          RequiredFields[FID] := Nullable = False;
          LoadFieldDef(Word(FID), FieldDescs[0]);
          if (FieldDescs[0].iLen > FMaxColSize) and
             (FieldDescs[0].iFldType <> fldBLOB) then
            FMaxColSize := (FMaxColSize + FieldDescs[0].iLen);
          AddFieldDesc(FieldDescs, Integer(0), FID, RequiredFields, FieldDefs);
        end;
      finally
        for FID := 1 to FldDescCount do
          FreeAndNil(FieldDescs[FID-1]);
        FieldDescs := nil;
      end;
    finally
      RequiredFields.Free;
    end;
  end
  else
     DatabaseError(SDataSetClosed, self);
end;

{ Field and Record Access }

procedure NormalizeBcdData(BcdData: PBcd; Precision, Scale: Word);
var
  ABcd: TBcd;
  Success: Boolean;
begin
  if Assigned(BcdData) then
  begin
    if Precision > MaxFMTBcdDigits then Precision := MaxFMTBcdDigits;
    if (BcdData.SignSpecialPlaces = 38) and ((Scale and 63)in [38,0]) then
    begin
      if (Scale and (1 shl 7)) <> 0 then
        Success := NormalizeBcd( BcdData^, ABcd, MaxFMTBcdDigits, Word((DefaultFMTBcdScale and 63) or (1 shl 7)))
      else
        Success := NormalizeBcd( BcdData^, ABcd, MaxFMTBcdDigits, DefaultFMTBcdScale);
    end else
      Success := NormalizeBcd( BcdData^, ABcd, Precision, Scale);
    if Success then
      BcdData^ := ABcd
    else
      DatabaseError(SBcdOverflow);
 end;
end;

function TCustomSQLDataSet.GetFieldData(FieldNo: Integer; Buffer: Pointer): Boolean;
var
  FldType, Subtype: Word;
  Status: SQLResult;
  FBlank: LongBool;
  Field: TField;
  Precision, Scale: Word;
begin
  if (FSQLCursor = nil) then
    DatabaseError(SDataSetClosed, self);

  {When EOF is True we should not be calling into the driver to get Data}
  if EOF and not BOF then /// fixed 157041
  begin
    Result := False;
    Exit;
  end;

  FBlank := True;
  Status := FSQLCursor.getColumnType(FieldNo, FldType, SubType);
  if (Status = 0) then
  begin
    case FldType of
      fldZSTRING:
        Status := FSQLCursor.GetString(FieldNo, Buffer, FBlank);
      fldWIDESTRING:
        Status := FSQLCursor.GetWideString(FieldNo, Buffer, FBlank);
      fldINT16, fldUINT16:
        Status := FSQLCursor.GetShort(FieldNo, Buffer, FBlank);
      fldINT32, fldUINT32:
        Status := FSQLCursor.GetLong(FieldNo, Buffer, FBlank);
      fldINT64:
        Status := FSQLCursor.GetInt64(FieldNo, Buffer, FBlank);
      fldFLOAT:
        Status := FSQLCursor.GetDouble(FieldNo, Buffer, FBlank);
      fldFMTBCD, fldBCD:
        begin
          Status := FSQLCursor.GetBcd(FieldNo, Buffer, FBlank);
          Field := FieldByNumber(FieldNo);
          if (not FBlank) and (Status = DBXERR_NONE) and (Field <> nil) then
          begin
            if Field.DataType = ftBcd then
            begin
              Precision := TBcdField(Field).Precision;
              Scale := TBcdField(Field).Size;
            end else
            begin
              Precision := TFMTBcdField(Field).Precision;
              Scale := TFMTBcdField(Field).Size;
            end;
            NormalizeBcdData(PBcd(Buffer), Precision, Scale);
          end;
        end;
      fldDATE:
        Status := FSQLCursor.GetDate(FieldNo, Buffer, FBlank);
      fldTIME:
        Status := FSQLCursor.GetTime(FieldNo, Buffer, FBlank);
      fldDATETIME:
        Status := FSQLCursor.GetTimeStamp(FieldNo, Buffer, FBlank);
      fldBOOL:
        Status := FSQLCursor.GetShort(FieldNo, Buffer, FBlank);
      fldBYTES, fldVARBYTES:
        Status := FSQLCursor.GetBytes(FieldNo, Buffer, FBlank);
      fldBLOB:
        begin
          GetBlobSize(Self, FieldNo);
          if CurrentBlobSize = 0 then
            FBlank := True
          else
            Status := FSQLCursor.GetBlob(FieldNo, Buffer, FBlank, CurrentBlobSize);
        end;
    end;
  end;
  Check(Status, exceptCursor);
  Result := not FBlank;
end;

function TCustomSQLDataSet.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
var
   FieldNo: Word;
   TempBuffer: PWideChar;
   ThisBuffer: Pointer;
   BlobSize: LongWord;
   BlobNull: LongBool;
begin
  if not Self.Active then
    DataBaseError(SDatasetClosed);
  FieldNo := Field.FieldNo;
  if not Assigned(Buffer) then
  begin
    if Field.IsBlob then
    begin
      if EOF then
        BlobNull := True
      else
        FSQLCursor.GetBlobSize(Word(FieldNo), BlobSize, BlobNull);
      Result := not Boolean(BlobNull);
      Exit;
    end
    else if Field.Size > Field.DataSize then
      TempBuffer := AllocMem(Field.Size)
    else
      TempBuffer := AllocMem(Field.DataSize);
    ThisBuffer := Pointer(TempBuffer);
  end else
  begin
    ThisBuffer := Buffer;
    TempBuffer := nil;
  end;
  try
    if Field.FieldNo < 1 then
      Result := GetCalculatedField(Field, ThisBuffer)
    else
      Result := GetFieldData(FieldNo, ThisBuffer);
  finally
    if Assigned(TempBuffer) then
      FreeMem(TempBuffer);
  end;
end;

procedure TCustomSQLDataSet.SetCurrentBlobSize(Value: LongWord);
begin
  FCurrentBlobSize := Value;
  SetLength(FBlobBuffer, FCurrentBlobSize);
end;

function TCustomSQLDataSet.GetBlobFieldData(FieldNo: Integer; var Buffer: TBlobByteData): Integer;
var
  IsNull: LongBool;
  FldType, SubType: Word;
begin
  Result := 0;
  GetBlobSize(Self, FieldNo);
  if (FSQLCursor = nil) then
    DatabaseError(SDataSetClosed, self);
  if FCurrentBlobSize > 0 then
  begin
    Check(FSQLCursor.getColumnType(LongWord(FieldNo), FldType, SubType), exceptCursor);
    if LongWord(Length(Buffer)) < CurrentBlobSize then
      SetLength(Buffer, CurrentBlobSize);
    if FCurrentBlobSize = 0 then
      Result := 0
    else
      Check(FSQLCursor.GetBlob(LongWord(FieldNo), PWideChar(Buffer), IsNull, FCurrentBlobSize), exceptCursor);
  end;
  if not IsNull then Result := CurrentBlobSize;
end;

function TCustomSQLDataSet.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
begin
  Result := TSQLBlobStream.Create(Field as TBlobField, Mode);
end;

procedure TCustomSQLDataSet.SetFieldData(Field: TField; Buffer: Pointer);
var
  RecBuf: PChar;
begin
  RecBuf := FCalcFieldsBuffer;
  with Field do
  begin
    if FieldNo < 1 then   //{fkCalculated}
    begin
      Inc(RecBuf, Offset);
      Boolean(RecBuf[0]) := LongBool(Buffer);
      if Boolean(RecBuf[0]) then Move(Buffer^, RecBuf[1], DataSize);
    end;
  end;
end;

function TCustomSQLDataSet.GetCalculatedField(Field: TField; var Buffer: Pointer): Boolean;
var
  RecBuf: PChar;
begin
  Result := False;
  RecBuf := FCalcFieldsBuffer;
  with Field do
  begin
    if FieldNo < 1 then   //{fkCalculated}
    begin
      Inc(RecBuf, Offset);
      if Boolean(RecBuf[0]) then
      begin
        Move(RecBuf[1], Buffer^, DataSize);
        Result := True;
      end;
    end;
  end;
end;

function TCustomSQLDataSet.GetCanModify: Boolean;
begin
  Result := False;
end;

function TCustomSQLDataSet.GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult;
var
  Status: SQLResult;
begin
  Status := FSQLCursor.next;
  if (not (Status in [DBXERR_NONE, SQL_NULL_DATA, DBXERR_EOF])) then
     Check(Status, exceptCursor);
  if Status = DBXERR_NONE then
  begin
    GetCalcFields(FCalcFieldsBuffer);
    Result := grOK
  end
  else
    Result := grEOF;
end;

{ CommandText Management }

procedure TCustomSQLDataSet.SetFCommandText(const Value: string);
begin
  CheckInactive;
  FCommandText := Value;
  FNativeCommand := '';
end;

procedure TCustomSQLDataSet.SetCommandText(const Value: WideString);
var
  HasDataLink: Boolean;
  DataSet: TDataSet;
begin
  if FCommandText <> Value then
  begin
    CheckInactive;
    PropertyChanged;
    FCommandText := Trim(Value);
    if (SQLConnection <> nil) and (Value <> '') then
    begin
      if FParamCheck and (FCommandType <> ctTable) then
      begin
        HasDataLink := (FDataLink.DataSource <> nil) and (FDataLink.DataSource.DataSet is TCustomSQLDataSet);
        if HasDataLink then
          DataSet := FDataLink.DataSource.DataSet
        else
          DataSet := nil;
        SetParamsFromSQL(DataSet, not HasDataLink);
      end;
    end;
    DataEvent(dePropertyChange, 0);
  end;
end;

function TCustomSQLDataSet.GetDataSetFromSQL(TableName: WideString): TCustomSQLDataSet;
var
  Q: WideString;
begin
  if TableName = '' then
    TableName := GetTableNameFromSQL(SSelectStarFrom +
              Copy(CommandText, 8, Length(CommandText) - 7));
  if TableName = '' then
    Result := nil
  else
  begin
    Result := TCustomSQLDataSet.Create(nil);
    try
      Result.SetConnection(Self.SQLConnection);
      Q := Self.FSqlConnection.GetQuoteChar;
      Result.CommandText := SSelectStarFrom +
                  Q + TableName + Q +
                  SWhere + ' 0 = 1';    // only metadata is needed
      Result.Active := True;
    except
      FreeAndNil(Result);
    end;
  end;
end;

{ Parameters }

function TCustomSQLDataSet.GetProcParams: TList;
begin
  if (Self.FSQLConnection.Connected) and not Assigned(FProcParams) then
  begin
    FProcParams := TList.Create;
    FSQLConnection.GetProcedureParams(CommandText, FSchemaInfo.PackageName, FSchemaName, FProcParams);
  end;
  Result := FProcParams;
end;

procedure TCustomSQLDataSet.SetParamsFromProcedure;
var
  List: TParams;
begin
  List := TParams.Create;
  try
    try
      { Preserve existing values }
      List.AssignValues(Params);
      if Assigned(FProcParams) then
        FreeProcParams(FProcParams);
      ProcParams := TList.Create;
      FSQLConnection.GetProcedureParams(CommandText, FSchemaInfo.PackageName, FSchemaName, ProcParams);
      LoadParamListItems(List, FProcParams);
    except
      FreeProcParams(FProcParams);
    end;
    if List.Count > 0 then
      Params.Assign(List);
  finally
    List.Free;
  end;
end;

procedure TCustomSQLDataSet.SetParamsFromSQL(DataSet: TDataSet; bFromFields: Boolean);
var
  Field: TField;
  I: Integer;
  List: TSQLParams;
  WasDatasetActive: Boolean;
  FTblName: WideString;
  DSCreated: Boolean;
begin
  DSCreated := False;
  FNativeCommand := Copy(CommandText, 1, Length(CommandText));
  if (CommandType = ctStoredProc) then
  begin
    SetParamsFromProcedure;
    Exit;
  end;
  List := TSQLParams.Create(Self);
  try                                              // DBExpress only supports '?', so
    FTblName := List.Parse(FNativeCommand, True);  // save query to avoid
    { Preserve existing values }                   // parsing again with prepare
    List.AssignValues(Params);
    if (Assigned(SQLConnection)) and (List.Count > 0) then
      begin
        WasDataSetActive := True;
        if DataSet = nil then
        begin
          if FTblName <> '' then
          begin
            if csDesigning in ComponentState then
            begin
              DataSet := GetDataSetFromSQL(FTblName);
              if Assigned(DataSet) then
                DSCreated := True;
            end;
          end;
        end else begin
          WasDataSetActive := DataSet.Active;
          if not DataSet.Active then DataSet.Active := True;
        end;
        for I := 0 to List.Count - 1 do
          List[I].ParamType := ptInput;
        if (DataSet <> nil) and
              ((not List.BindAllFields) or
              (List.Count = DataSet.FieldCount)) then
          try
            for I := 0 to List.Count - 1 do
            begin
              if List.BindAllFields then
                Field := DataSet.Fields[I]
              else if List.FFieldName.Count > I then
              begin
                if (bFromFields) then
                  Field := DataSet.FieldByName(List.GetFieldName(I))
                else
                  Field := DataSet.FieldByName(List[I].Name);
              end else
                 Field := nil;
              if Assigned(Field) then
              begin
                if Field.DataType <> ftString then
                  List[I].DataType := Field.DataType
                else if TStringField(Field).FixedChar then
                  List[I].DataType := ftFixedChar
                else
                  List[I].DataType := ftString;
              end;
            end;
          except
            // ignore exception: Column type won't be provided
          end;
        if List.Count > 0 then
          Params.Assign(List);
        if Assigned(DataSet) and (not WasDataSetActive) then DataSet.Active := False;
      end
    else
      Params.clear;
  finally
    List.Free;
    if DSCreated then DataSet.Free;
  end;
end;

procedure TCustomSQLDataSet.RefreshParams;
var
  DataSet: TDataSet;
begin
  DisableControls;
  try
    if FDataLink.DataSource <> nil then
    begin
      DataSet := FDataLink.DataSource.DataSet;
      if DataSet <> nil then
        if DataSet.Active and (DataSet.State <> dsSetKey) then
        begin
          Close;
          Open;
        end;
    end;
  finally
    EnableControls;
  end;
end;

procedure TCustomSQLDataSet.SetParamsFromCursor;
var
  I: Integer;
  DataSet: TDataSet;
begin
  if (FDataLink.DataSource <> nil) and (FParams.Count > 0) then
  begin
    DataSet := FDataLink.DataSource.DataSet;
    if (DataSet <> nil) then
    begin
      for I := 0 to FParams.Count - 1 do
        with FParams[I] do
          if not Bound then
          begin
            if not DataSet.eof then
              AssignField(DataSet.FieldByName(Name))
            else
              FParams[I].Value := Null;
            Bound := False;
          end;
    end;
  end;
end;

function TCustomSQLDataSet.ParamByName(const Value: string): TParam;
begin
  Result := FParams.ParamByName(Value);
end;

procedure TCustomSQLDataSet.GetOutputParams(AProcParams: TList);
var
  I: Integer;
  RecBuffer: PWideChar;
  ArgDesc: SPParamDesc;
  IsNull: Integer;
begin
  for I := 0 to Params.Count - 1 do
  begin
    isNull := 0;
    if AProcParams <> nil then
      ArgDesc := (PSPParamDesc(AProcParams.Items[I]))^
    else
      with ArgDesc, Params[i] do
        begin
          iParamNum := i + 1;
          szName := Name;
          iArgType := ParamType;
          iDataType := DataType;
          iUnits1 := Precision;
          iUnits2 := NumericScale;
          iLen := GetDataSize;
        end;
    if (Params[I].ParamType in [ptOutput, ptResult, ptInputOutput]) and
       (ArgDesc.iDataType <> ftCursor) then
    begin
      RecBuffer := AllocMem(ArgDesc.iLen + 1);
      try
        Check(FSQLCommand.getParameter(i + 1, 0, RecBuffer, ArgDesc.iLen, IsNull), exceptCommand);
        if IsNull = 0 then
          Params[I].SetData(RecBuffer)
        else
          Params[I].Value := Null;
      finally
        FreeMem(RecBuffer);
      end;
    end;
  end;
end;

procedure TCustomSQLDataSet.SetParameters(const Value: TParams);
begin
  FParams.AssignValues(Value);
end;

{ Query Management }

procedure TCustomSQLDataSet.SetPrepared(Value: Boolean);
begin
  if Value then CheckConnection(eConnect);
  if FGetNextRecordSet then
    FPrepared := Value
  else
    FreeCursor;
  if SchemaInfo.FType <> stNoSchema then
  begin
    if Value then
      CheckStatement(True)
    else
      FreeStatement;
  end
  else
  if Value <> Prepared then
  begin
    try
      if Value then
        begin
          if FSQLCommand <> nil then DatabaseError(SSQLDataSetOpen, Self);
          FRowsAffected := -1;
          FCheckRowsAffected := True;
          PrepareStatement;
        end
      else
        begin
          if FCheckRowsAffected then
            FRowsAffected := RowsAffected;
          FreeStatement;
          if Assigned(FSQLConnection) then
            FSQLConnection.CheckDisconnect;
        end;
      FPrepared := Value;
    except
      if Assigned(FSQLCommand) then
        FreeStatement;
      FPrepared := False;
    end;
  end;
end;

procedure TCustomSQLDataSet.CheckStatement(ForSchema: Boolean = False);
var
  Connection: TSqlConnection;
  RowsetSize: Integer;
begin
  FLastError := '';
  RowsetSize := defaultRowsetSize;
  if not Assigned(FSQLConnection) then
    DatabaseError(SMissingSQLConnection);
  Connection := FSQLConnection.GetConnectionForStatement;
  if Connection.FIsCloned then
    FClonedConnection := Connection;
  if Connection.LoadParamsOnConnect then
    Connection.LoadParamsFromIniFile;
  if Assigned(FSQLCommand) then
    FreeStatement;
  if not Assigned(Connection.Connection) then
    DatabaseError(SdatabaseOpen, Self);
  if not ForSchema then
  begin
    if Length(FCommandText) = 0 then
      DatabaseError(SEmptySQLStatement, Self);
    Check(Connection.Connection.getSQLCommand(FSQLCommand), exceptCommand);

    if FSQLConnection.Params.Values[ROWSETSIZE_KEY] <> '' then
    try
      RowsetSize := StrToInt(trim(FSQLConnection.Params.Values[ROWSETSIZE_KEY]));
    except
      RowsetSize := defaultRowsetSize;
    end;

	FSQLCommand.setOption(eCommRowsetSize, RowsetSize);

    FStatementOpen := True;
    if FTransactionLevel > 0 then
      FSQLCommand.SetOption(eCommTransactionID, Integer(FTransactionLevel));
    if FNativeCommand = '' then
    begin
      if FParams.Count > 0 then
        FNativeCommand := FixParams(CommandText, FParams.Count, Connection.GetQuoteChar)
      else
        FNativeCommand := CommandText;
    end;
  end;
end;

function TCustomSQLDataSet.GetQueryFromType: WideString;
var
  STableName : String;
begin
  case CommandType of
     ctTable:
       begin
         if Self.FSchemaName <> '' then
           STableName := AddQuoteCharToObjectName(Self, FSchemaName + '.' + FCommandText)
         else
           STableName := AddQuoteCharToObjectName(Self, FCommandText);
         if FSortFieldNames > '' then
           Result := SSelectStarFrom + STableName + SOrderBy + FSortFieldNames
         else
           if FNativeCommand = '' then
             Result := SSelectStarFrom + STableName
           else
           begin
             if Trim(FSchemaName) <> '' then
               STableName := AddQuoteCharToObjectName(Self, FSchemaName + '.' + FNativeCommand)
             else
              STableName := AddQuoteCharToObjectName(Self, FNativeCommand);
             Result := SSelectStarFrom + STableName;
           end;
       end;
     ctStoredProc:
       begin
         if FSchemaName <> '' then
	   Result := AddQuoteCharToObjectName(Self, FSchemaName + '.' + FCommandText)
         else
           Result := AddQuoteCharToObjectName(Self, FCommandText)
       end;
     else
       if (FSortFieldNames > '') and (Pos(SOrderBy,
          WideLowerCase(FCommandText)) = 0) then
         Result := FNativeCommand + SOrderBy + FSortFieldNames
       else
         Result := FNativeCommand;
  end;
end;

function TCustomSQLDataSet.CheckDetail(const SQL: WideString): WideString;
begin
  Result := SQL;
  if pos(SParam, SQL) = 0 then
    if pos(SSelect, WideLowerCase(SQL)) > 0 then // Select Query with no ?, but Parameters are set
      Result := AddParamSQLForDetail(Params, SQL, True);
end;

procedure TCustomSQLDataSet.PrepareStatement;
var
  SQLText, Value: WideString;
  Command: pWideChar;
  CurSection : TSqlToken;
begin
  if Length(CommandText) = 0 then
    DatabaseError(SEmptySQLStatement, Self);
  Command := PWideChar(CommandText);
  CurSection := stUnknown;
  CurSection := NextSQLToken(Command, Value, CurSection);
  if (CurSection = stSelect) or (CommandType = ctTable) then
    Inc(FSQLConnection.FSelectStatements);
  CheckStatement;
  SQLText := GetQueryFromType;
  CurSection := NextSQLToken(Command, Value, CurSection);
  if (CurSection = stSelect) or (CommandType = ctTable) then
    Inc(FSQLConnection.FSelectStatements);
  if Params.Count > 0 then
    SQLText := CheckDetail(SQLText);
  if CommandType = ctStoredProc then
    Check(FSQLCommand.SetOption(eCommStoredProc, Integer(True)), exceptCommand)
  else
    Check(FSQLCommand.SetOption(eCommStoredProc, Integer(False)), exceptCommand);
  Check(FSQLCommand.prepare(PWideChar(WideString(SQLText)), ParamCount), exceptCommand);
end;

procedure TCustomSQLDataSet.CheckPrepareError;
begin
  if (FSQLCommand = nil) and (SchemaInfo.FType = stNoSchema) then
  begin     // prepare has failed
    if FLastError <> '' then
      SQLError(0, exceptUseLast)
    else if (CommandType = ctQuery) or (SortFieldNames <> '') then
      DatabaseError(sPrepareError)
    else
      DatabaseError(sObjectNameError);
  end;
end;

function TCustomSQLDataSet.ExecSQL(ExecDirect: Boolean = False): Integer;
begin
  CheckInActive;
  CheckConnection(eConnect);
  try
    FRowsAffected := 0;
    if not ExecDirect then
    begin
      SetPrepared(True);
      CheckPrepareError;
      ExecuteStatement;
    end else
    begin
      CheckStatement;
      Check(FSQLCommand.executeImmediate(PWideChar(CommandText), FSQLCursor), exceptCommand);
    end;
    if FSQLCursor <> nil then
       SetParamsFromCursor;
    Result := RowsAffected;
  finally
    if Assigned(FSQLCursor) then
    begin
      FreeCursor;
      FreeStatement;
    end else if ExecDirect then
      FreeStatement
    else
      CloseStatement;
  end;
end;

procedure TCustomSQLDataSet.ExecuteStatement;

 function UseParams(): Boolean;
  const
    SDelete = 'delete';      { Do not localize }
    SUpdate = 'update';      { Do not localize }
    SInsert = 'insert';      { Do not localize }
  var
    SQL: WideString;
  begin
    Result := (FParams.Count <> 0);
    if Result and (FCommandType = ctTable) then
    begin
       if FNativeCommand <> ''  then
         SQL := FNativeCommand
       else
         SQL := FCommandText;
       Result := SqlRequiresParams(SQL);
    end;
  end;


var
  Status: SQLResult;
begin
  if SchemaInfo.FType = stNoSchema then
    begin
      if Assigned(FParams) and not FGetNextRecordSet then
      begin
        if CommandType = ctStoredProc then
          SetQueryProcParams(Self.FSQLConnection, FSQLCommand, Params, ProcParams)
        else
        if UseParams() then
          SetQueryProcParams(Self.FSQLConnection, FSQLCommand, Params);
      end;
      if FGetNextRecordSet then
      begin
        Status := FSQLCommand.getNextCursor(FSQLCursor);
        if not (Status in [DBXERR_NONE, DBXERR_EOF, SQL_NULL_DATA]) then
          Check(Status, exceptCommand);
        if Status <> DBXERR_NONE then
        begin
          if Active then
            Active := False
          else  // Active might be false when calling getNextCursor.
            CloseCursor;
        end else
          begin
            if CommandType = ctStoredProc then
              begin
                if Params.Count > 0 then
                  GetOutputParams(FProcParams);
              end
            else
              begin
                if Params.Count > 0 then
                  GetOutputParams;
              end;
          end;
      end
      else
      begin
        Check(FSQLCommand.execute(FSQLCursor), exceptCommand);
        if CommandType = ctStoredProc then
          begin
            if Params.Count > 0 then
              GetOutputParams(FProcParams);
          end
        else
          begin
            if Params.Count > 0 then
              GetOutputParams;
          end;
      end;
    end
  else
    OpenSchema;
  FStatementOpen := True;
  FRecords := -1;
end;

function TCustomSQLDataSet.GetObjectProcParamCount: Integer;
var
  I, LastParamNum: Integer;
  ArgDesc: SPParamDesc;
begin
  GetProcParams;    // make sure FProcParams is loaded.
  Result := 0;
  LastParamNum := 0;
  for I := 0 to Params.Count -1 do
  begin
    ArgDesc := (PSPParamDesc(ProcParams.Items[I]))^;
    if ArgDesc.iParamNum <> LastParamNum then Inc(Result);
    LastParamNum := ArgDesc.iParamNum;
  end;
end;

function TCustomSQLDataSet.GetParamCount: Integer;
var
  I : Integer;
begin
  Result := FParamCount;
  if Result = -1 then
  begin
    Result := 0;
    if Assigned(FParams) then
    begin
      if FCommandType = ctStoredProc then
      begin
        for I := 0 to Params.Count -1 do
        begin
          if Params.Items[I].DataType in [ftADT, ftARRAY] then
          begin
            Result := GetObjectProcParamCount;
            break;
          end;
        end;
      end;
      if Result = 0 then Result := FParams.Count
    end;
  end;
end;

function GetRows(Query: string; Connection: TSQLConnection): Integer;
var
  DS: TSQLDataSet;
begin
  Result := -1;
  DS := TSQLDataSet.Create(nil);
  try
    DS.SQLConnection := Connection;
    DS.CommandText := Query;
    DS.Active := True;
    if not DS.EOF then
      Result := DS.Fields[0].AsInteger;
  finally
    DS.Free;
    if Result = -1 then
      DatabaseError(SNotSupported);
  end;
end;

function TCustomSQLDataSet.GetRecordCount: Integer;
const
  SDistinct = ' distinct ';                 { do not localize }
  SSelectCount = 'select count(*) from ';   { do not localize }
var
  TableName, Query: string;
  HoldPos: Integer;
  Status : SQLResult;
  str : WideString;
begin
  if FRecords <> -1 then
    Result := FRecords
  else
  begin
    CheckConnection(eConnect);
    if Self.CommandText = '' then
      DatabaseError(SNoSQLStatement);
    case CommandType of
      ctStoredProc:
        DatabaseError(SNotSupported);
      ctTable:
        begin
          //Query := 'select count(*) from ' + GetQuoteChar + FCommandText + GetQuoteChar;
          Status := GetInternalConnection.FISQLConnection.setStringOption(eConnQualifiedName, CommandText);
          if Status <> 0 then
            SQLError(Status, exceptConnection);
          SetLength(str, 256);
          Status := GetInternalConnection.FISQLConnection.getStringOption(eConnQuotedObjectName, str);
          if Status <> 0 then
            SQLError(Status, exceptConnection);
          Query := 'select count(*) from ' + str;
        end;
      ctQuery:
        begin
          TableName := GetTableNameFromSQL(FCommandText);
          if (TableName = '') or (Params.Count > 0) then
            DatabaseError(SNotSupported);
          if Pos(SDistinct, WideLowerCase(FCommandText)) = 0 then
            Query := SSelectCount
          else
            DatabaseError(SNotSupported);
          HoldPos := Pos(SWhere, WideLowerCase(FCommandText));
          if HoldPos = 0 then
            Query := Query + GetQuoteChar + TableName + GetQuoteChar
          else begin
            Query := Query + GetQuoteChar + TableName + GetQuoteChar + copy(FCommandText, HoldPos, Length(FCommandText) - (HoldPos-1));
            HoldPos := Pos(sOrderBy, WideLowerCase(Query));
            if HoldPos > 0 then
              Query := copy(Query, 1, HoldPos - 1);
          end;
        end;
    end;
    FRecords := GetRows(Query, FSQLConnection);
    Result := FRecords;
  end;
end;

function TCustomSQLDataSet.GetRowsAffected: Integer;
var
  UpdateCount: LongWord;
begin
  if FRowsAffected > 0 then
    Result := Integer(FRowsAffected)
  else
    begin
      if FSQLCommand <> nil then
        Check(FSQLCommand.getRowsAffected(UpdateCount), exceptCommand)
      else
        UpdateCount := 0;
      FRowsAffected := Integer(UpdateCount);
      Result := Integer(UpdateCount);
    end;
end;

{ Misc. Set/Get Property }

procedure TCustomSQLDataSet.SetDataSource(Value: TDataSource);
begin
  if IsLinkedTo(Value) then DatabaseError(SCircularDataLink, Self);
  if FDataLink.DataSource <> Value then
    FDataLink.DataSource := Value;
end;

function TCustomSQLDataSet.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCustomSQLDataSet.GetDetailLinkFields(MasterFields, DetailFields: TList);

  function AddFieldToList(const FieldName: string; DataSet: TDataSet;
    List: TList): Boolean;
  var
    Field: TField;
  begin
    Field := DataSet.FindField(FieldName);
    if Field <> nil then
      List.Add(Field);
    Result := Field <> nil;
  end;

var
  I: Integer;
begin
  MasterFields.Clear;
  DetailFields.Clear;
  if (DataSource <> nil) and (DataSource.DataSet <> nil) then
    for I := 0 to Params.Count - 1 do
      if AddFieldToList(Params[I].Name, DataSource.DataSet, MasterFields) then
        AddFieldToList(Params[I].Name, Self, DetailFields);
end;

function TCustomSQLDataSet.GetSortFieldNames: WideString;
begin
  Result := FSortFieldNames;
end;

procedure TCustomSQLDataSet.SetSortFieldNames(Value: WideString);
begin
  FSortFieldNames := Value;
end;

procedure TCustomSQLDataSet.SetMaxBlobSize(MaxSize: Integer);
begin
  FMaxBlobSize := MaxSize;
  if (FSQLCommand <> nil) then
    FSQLCommand.SetOption(eCommBlobSize, Integer(MaxSize));
end;

procedure TCustomSQLDataSet.SetCommandType(const Value: TSQLCommandType);
begin
  if FCommandType <> Value then
  begin
    CheckInactive;
    FCommandType := Value;
    PropertyChanged;
    DataEvent(dePropertyChange, 0);
  end;
end;

procedure TCustomSQLDataSet.PropertyChanged;
begin
  if not (csLoading in ComponentState) then
  begin
    SetPrepared(False);
    FNativeCommand := '';
    FRecords := -1;
    FreeStatement;
    if SortFieldNames <> '' then
      FSortFieldNames := '';
    if FCommandText <> '' then
      FCommandText := '';
    FParams.Clear;
  end;
end;

{ Miscellaneous }

function TCustomSQLDataSet.IsSequenced: Boolean;
begin
  Result := False;
end;

procedure TCustomSQLDataSet.DefineProperties(Filer: TFiler);

  function DesignerDataStored: Boolean;
  begin
    if Filer.Ancestor <> nil then
      Result := TCustomSQLDataSet(Filer.Ancestor).DesignerData <> DesignerData else
      Result := DesignerData <> '';
  end;

begin
  inherited;
  Filer.DefineProperty('DesignerData', ReadDesignerData, WriteDesignerData,
    DesignerDataStored);
end;

procedure TCustomSQLDataSet.ReadDesignerData(Reader: TReader);
begin
  FDesignerData := Reader.ReadString;
end;

procedure TCustomSQLDataSet.WriteDesignerData(Writer: TWriter);
begin
  Writer.WriteString(FDesignerData);
end;

{ Exception Handling }

function TCustomSQLDataSet.Check(Status: SQLResult; eType: TSQLExceptionType): SQLResult;
begin
  if Status <> 0 then SQLError(Status, eType);
  Result := Status;
end;

procedure TCustomSQLDataSet.InternalHandleException;
begin
end;

{ Index Support }

procedure TCustomSQLDataSet.UpdateIndexDefs;
begin
  AddIndexDefs(Self);
end;

function TCustomSQLDataSet.CheckFieldNames(const FieldNames: WideString): Boolean;
var
  S: WideString;
  Pos: Integer;
begin
  Result := True;
  S := FieldNames;
  Pos := 1;
  while Result and (Pos <= Length(S)) do
    Result := FindField(ExtractFieldName(S, Pos)) <> nil;
end;

const
  IDX_TYPE_FIELD = 'INDEX_TYPE';           { Do not localize }
  IDX_SORT_FIELD = 'SORT_ORDER';           { Do not localize }
  DescendingOrder = 'D';                   { Do not localize }

procedure TCustomSQLDataSet.AddIndexDefs(SourceDS: TCustomSQLDataSet; IndexName: string = '');

  function DontUseIndex: Boolean;
  begin
    Result := CommandType in [ctQuery, ctStoredProc];
    if Result and (CommandType = ctQuery) then
      Result := IsMultiTableQuery(CommandText);
    if Result then FIndexDefsLoaded := True;
  end;

var
  DataSet: TCustomSQLDataSet;
  TableName, IdxName, SortOrder, FieldNames: string;
  IdxType: Integer;
  Options: TIndexOptions;
  IdxDef: TIndexDef;
begin
  if not FGetMetadata then FIndexDefsLoaded := True;
  if FIndexDefsLoaded then Exit;
  if SchemaInfo.FType <> stNoSchema then Exit;
  if DontUseIndex then Exit;
  if FCommandType = ctTable then
    TableName := FCommandText
  else
    TableName := GetTableNameFromSQL(CommandText);
  DataSet := FSQLConnection.OpenSchemaTable(stIndexes, TableName, '', '', '');
  if not Assigned(DataSet) then FSQLConnection.SQLError(SQLResult(-1), exceptMetadata);
  try
    FIndexDefs.Clear;
    IndexDefs.Clear;
    while not DataSet.EOF do
    begin
      begin
        Options := [];
        IdxName := DataSet.FieldByName(IDX_NAME_FIELD).Value;
        if (IndexName = '') or (IdxName = IndexName) then
        begin
          if IndexDefs.IndexOf(IdxName) = -1 then
          begin
            FieldNames := DataSet.FieldByName(COL_NAME_FIELD).Value;
            // don't add indexes on fields not in result set
            if SourceDS.CheckFieldNames(FieldNames) then
            begin
              IdxType := DataSet.FieldByName(IDX_TYPE_FIELD).Value;
              if (IdxType and eSQLPrimaryKey) = eSQLPrimaryKey then
                Options := Options + [ixPrimary];
              if (IdxType and eSQLUnique) = eSQLUnique then
                Options := Options + [ixUnique];
              SortOrder := DataSet.FieldByName(IDX_SORT_FIELD).Value;
              if SortOrder = DescendingOrder then
                Options := Options + [ixDescending];
              FIndexDefs.Add(IdxName, FieldNames, Options);
            end;
          end else
          begin
            IdxDef := IndexDefs.Find(IdxName);
            IdxDef.Fields := IdxDef.Fields + ';' + DataSet.FieldByName(COL_NAME_FIELD).Value;
          end;
        end;
      end;
      DataSet.Next;
    end;
  finally
    FSQLConnection.FreeSchemaTable(DataSet);
  end;
  FIndexDefsLoaded := True;
end;

function TCustomSQLDataSet.GetKeyFieldNames(List: TStrings): Integer;
var
  wList: TWideStrings;
begin
  wList := TWideStringList.Create;
  try
    Result := GetKeyFieldNames(wList);
    List.Assign(wList);
  finally
    wList.Free
  end;
end;

function TCustomSQLDataSet.GetKeyFieldNames(List: TWideStrings): Integer;
var
  I: Integer;
begin
  if not FIndexDefsLoaded then
    AddIndexDefs(Self);
  Result := IndexDefs.Count;
  List.BeginUpdate;
  try
    List.Clear;
    for I := 0 to Result - 1 do
      List.Add(IndexDefs[I].Fields);
  finally
    List.EndUpdate;
  end;
end;

{ Schema Tables }

procedure TCustomSQLDataSet.SetSchemaInfo(SchemaType: TSchemaType; SchemaObjectName, SchemaPattern: WideString; PackageName: WideString = '' );
begin
  FreeStatement;
  FSchemaInfo.FType := SchemaType;
  FSchemaInfo.ObjectName := SchemaObjectName;
  FSchemaInfo.Pattern := SchemaPattern;
  FSchemaInfo.PackageName := PackageName;
end;

procedure TCustomSQLDataSet.OpenSchema;

  function ExtractObjectName(Value: WideString): WideString;
  var
    NamePos: Integer;
    Q: WideString;
  begin
    Result := Value;
    Q := GetQuoteChar;
    if (Q = '') or (Q = ' ') then exit;
    NamePos := Pos('.' + Q, Value);
    if NamePos = 0 then
      NamePos := Pos(Q + '.', Value);
    if NamePos = 0 then exit;
    Result := Copy(Value, NamePos + 2, Length(Value) - NamePos);
    if Pos(Q, Result) > 0 then
      Result := Copy(Result, 1, Length(Result) -1);
  end;

var
  Status: SQLResult;
  TblType: LongWord;
  WildCard: PWideChar;
  SaveSchema: WideString;
begin
  Status := SQL_NULL_DATA;
  if FSQLConnection = nil then
    DatabaseError(sConnectionNameMissing);
  if FSchemaInfo.Pattern = '' then
    WildCard := nil
  else
    WildCard := PWideChar(FSchemaInfo.Pattern);
  SetSchemaOption;
  //FSchemaInfo.ObjectName := ExtractObjectName(FSchemaInfo.ObjectName);
  if Assigned(FSQLCursor) then
    FreeAndNil(FSQLCursor);
  case FSchemaInfo.FType of
    stTables:
    begin
      TblType := GetTableScope(GetInternalConnection.FTableScope);
      Status := GetInternalConnection.FSQLMetaData.getTables(
                  WildCard, TblType, FSQLCursor)
    end;
    stSysTables:
      Status := GetInternalConnection.FSQLMetaData.getTables(
                  WildCard, eSQLSystemTable, FSQLCursor);
    stColumns:
      Status := GetInternalConnection.FSQLMetaData.getColumns(
                  PWideChar(FSchemaInfo.ObjectName),
                  PWideChar(FSchemaInfo.Pattern), 0, FSQLCursor);
    stProcedures:
    begin
      Status := GetInternalConnection.FSQLMetaData.setStringOption(
                  eMetaPackageName, FSchemaInfo.PackageName);
      if Status = DBXERR_NONE then
        Status := GetInternalConnection.FSQLMetaData.getProcedures(
                    WildCard, (eSQLProcedure or eSQLFunction), FSQLCursor);
    end;
    stPackages:
      Status := GetInternalConnection.FSQLMetaData.getObjectList(
                  eObjTypePackage, FSQLCursor);

    stUserNames:
      Status := GetInternalConnection.FSQLMetaData.getObjectList(
                  eObjTypeUser, FSQLCursor);

    stProcedureParams:
    begin
      Status := GetInternalConnection.FSQLMetaData.setStringOption(
                  eMetaPackageName, FSchemaInfo.PackageName);
      if Status = DBXERR_NONE then
        Status := GetInternalConnection.FSQLMetaData.getProcedureParams(
                    PWideChar(FSchemaInfo.ObjectName), WildCard,
                    FSQLCursor);
    end;
    stIndexes:
    begin
      if FSchemaInfo.Pattern <> '' then
      begin
        SetLength(SaveSchema, 128);
        Status := GetInternalConnection.FSQLMetaData.GetStringOption(eMetaSchemaName, SaveSchema);
        if Status = DBXERR_NONE then
          GetInternalConnection.FSQLMetaData.SetStringOption(eMetaSchemaName, FSchemaInfo.Pattern);
      end else
        Status := DBXERR_NONE;
      try
        if Status = DBXERR_NONE then
        begin
          Status := GetInternalConnection.FSQLMetaData.getIndices(
                  PWideChar(FSchemaInfo.ObjectName), 0, FSQLCursor);
        end;
      finally
        if FSchemaInfo.Pattern <> '' then
          GetInternalConnection.FSQLMetaData.SetStringOption(eMetaSchemaName, SaveSchema);
      end;
    end;
  end;
  if Status <> DBXERR_NONE then
    GetInternalConnection.SQLError(Status, exceptMetaData);
end;

{ ProviderSupport }

procedure TCustomSQLDataSet.PSEndTransaction(Commit: Boolean);
var TransDesc: TTransactionDesc;
begin
   FillChar(TransDesc, Sizeof(TransDesc), 0);
   TransDesc.TransactionID := 1;
   TransDesc.IsolationLevel := FSQLConnection.FTransIsoLevel;
   if Commit then FSQLConnection.Commit(TransDesc)
   else FSQLConnection.Rollback(TransDesc);
end;

procedure TCustomSQLDataSet.PSExecute;
begin
   ExecSQL;
end;

function TCustomSQLDataSet.PSExecuteStatement(const ASQL: WideString; AParams: TParams;
  ResultSet: Pointer = nil): Integer;
begin
  if Assigned(ResultSet) then
    Result := FSQLConnection.execute(ASQL, AParams, ResultSet)
  else
    Result := FSQLConnection.execute(ASQL, AParams);
end;

procedure TCustomSQLDataSet.PSGetAttributes(List: TList);
var
  Attr: PPacketAttribute;
begin
  inherited PSGetAttributes(List);
  New(Attr);
  List.Add(Attr);
  with Attr^ do
  begin
    Name := SLocaleCode;
    Value := Integer(FSQLConnection.LocaleCode);
    IncludeInDelta := False;
  end;
end;

function TCustomSQLDataSet.PSGetIndexDefs(IndexTypes: TIndexOptions): TIndexDefs;
begin
  if (not FIndexDefsLoaded) and (CommandType <> ctStoredProc)
     and (SchemaInfo.FType = stNoSchema) then
    AddIndexDefs(Self);
  Result := GetIndexDefs(IndexDefs, IndexTypes);
end;

function TCustomSQLDataSet.PSGetDefaultOrder: TIndexDef;

  function FieldsInQuery(IdxFields: string): Boolean;
  var
    I:  Integer;
    IdxFlds, Flds: TWideStrings;
    FldNames: string;
  begin
    Result := True;
    IdxFlds := TWideStringList.Create;
    try
      IdxFlds.CommaText := IdxFields;
      Flds := TWideStringList.Create;
      try
        Fields.GetFieldNames(Flds);
        FldNames := Flds.CommaText;
        for I := 0 to IdxFlds.Count -1 do
        begin
          if pos(IdxFlds[I], FldNames) = 0 then
          begin
            Result := False;
            exit;
          end;
        end;
      finally
        Flds.Free;
      end;
    finally
      IdxFlds.Free;
    end;
  end;

var
  I: Integer;
begin
  Result := inherited PSGetDefaultOrder;
  if not Assigned(Result) then
    Result := GetIndexForOrderBy(GetQueryFromType, Self);
  if (not Assigned(Result)) and
     (CommandType <> ctStoredProc) and (SchemaInfo.FType = stNoSchema) then
  begin
    if not FIndexDefsLoaded then
      AddIndexDefs(Self);
    for I := 0 to IndexDefs.Count - 1 do
    begin
      if (ixPrimary in TIndexDef(IndexDefs[I]).Options) and
         FieldsInQuery(TIndexDef(IndexDefs[I]).Fields) then
      begin
        Result := TIndexDef.Create(nil);
        Result.Assign(IndexDefs[I]);
        Break;
      end;
    end;
  end;
end;

function TCustomSQLDataSet.PSGetKeyFieldsW: WideString;
var
  HoldPos, I: Integer;
  IndexFound:Boolean;
begin
  if (CommandType = ctStoredProc) or (SchemaInfo.FType <> stNoSchema) then exit;
  Result := inherited PSGetKeyFieldsW;
  IndexFound := False;
  if (Result = '') and (SchemaInfo.FType = stNoSchema) then
  begin
    if not FIndexDefsLoaded then
      AddIndexDefs(Self);
    for I := 0 to IndexDefs.Count - 1 do
      if (ixUnique in IndexDefs[I].Options) or
         (ixPrimary in IndexDefs[I].Options) then
      begin
        Result := IndexDefs[I].Fields;
        IndexFound := (FieldCount = 0);
        if not IndexFound then
        begin
          HoldPos := 1;
          while HoldPos <= Length(Result) do
          begin
            IndexFound := FindField(ExtractFieldName(Result, HoldPos)) <> nil;
            if not IndexFound then Break;
          end;
        end;
        if IndexFound then Break;
      end;
    if not IndexFound then
      Result := '';
  end;
end;

function TCustomSQLDataSet.PSGetParams: TParams;
begin
  Result := Params;
end;

function TCustomSQLDataSet.GetQuoteChar: WideString;
begin
  Result := PSGetQuoteCharW;
end;

function TCustomSQLDataSet.PSGetQuoteCharW: WideString;
begin
  Result := '';
  if (Assigned(FSqlConnection) and (FSQLConnection.QuoteChar <> '')) then
    Result := FSQLConnection.QuoteChar;
end;

procedure TCustomSQLDataSet.PSReset;
begin
  inherited PSReset;
  if Active and (not BOF) then
    First;
end;

function TCustomSQLDataSet.PSGetTableNameW: WideString;
begin
   if CommandType = ctTable then
     Result := CommandText
   else
     Result := GetTableNameFromSQL(CommandText);
end;

function TCustomSQLDataSet.PSGetUpdateException(E: Exception; Prev: EUpdateError): EUpdateError;
begin
  if not Assigned(E) then
    E := EDatabaseError.Create(SErrorMappingError);
  Result := inherited PSGetUpdateException(E, Prev);
end;

function TCustomSQLDataSet.PSInTransaction: Boolean;
begin
  Result := (FSQLConnection <> nil) and (FSQLConnection.InTransaction);
end;

function TCustomSQLDataSet.PSIsSQLBased: Boolean;
begin
  Result := True;
end;

function TCustomSQLDataSet.PSIsSQLSupported: Boolean;
begin
  Result := True;
end;

procedure TCustomSQLDataSet.PSSetParams(AParams: TParams);
begin
  if (AParams.Count <> 0) and (AParams <> Params) then
  begin
    Params.Assign(AParams);
    if Prepared and (pos(SParam, FNativeCommand) = 0) then
      SetPrepared(False);
  end;
  Close;
end;

procedure TCustomSQLDataSet.PSSetCommandText(const ACommandText: WideString);
begin
  if ACommandText <> '' then
    CommandText := ACommandText;
end;

procedure TCustomSQLDataSet.PSStartTransaction;
var TransDesc: TTransactionDesc ;
begin
  FillChar(TransDesc, Sizeof(TransDesc), 0);
  TransDesc.TransactionID := 1;
  TransDesc.IsolationLevel := FSQLConnection.FTransIsoLevel;
  FSQLConnection.StartTransaction(TransDesc);
end;

function TCustomSQLDataSet.PSUpdateRecord(UpdateKind: TUpdateKind; Delta: TDataSet): Boolean;
begin
  { OnUpdateRecord is not supported }
  Result := False;
end;

function TCustomSQLDataSet.PSGetCommandText: string;
begin
  Result := CommandText;
end;

function TCustomSQLDataSet.PSGetCommandType: TPSCommandType;
begin
  Result := CommandType;
end;

function TCustomSQLDataSet.LocateRecord(const KeyFields: string; const KeyValues: Variant;
  Options: TLocateOptions; SyncCursor: Boolean): Boolean;

  function SameValue(V1, V2: Variant; IsString, CaseInsensitive,
           PartialLength: Boolean): Boolean;
  var
    V: Variant;
  begin
    if not IsString then
      Result := VarCompareValue(V1, V2) = vrEqual
    else
    begin
      if PartialLength then
        V := Copy(V1, 1, Length(V2))                                          
      else
        V := V1;
      if CaseInsensitive then
        Result := WideLowerCase(V) = WideLowerCase(V2)
      else
        Result := V = V2;
    end;
  end;

  function CheckValues(AFields: TStrings; Values: Variant;
           CaseInsensitive, PartialLength: Boolean): Boolean;
  var
    J: Integer;
    Field: TField;
  begin
    Result := True;
    for J := 0 to AFields.Count -1 do
    begin
      Field := FieldByName(AFields[J]);
      if not SameValue(Field.Value, Values[J],
        Field.DataType in [ftString, ftFixedChar], CaseInsensitive, PartialLength) then
      begin
        Result := False;
        break;
      end;
    end;
  end;

var
  I: Integer;
  SaveFields, AFields: TStrings;
  PartialLength, CaseInsensitive: Boolean;
  Values, StartValues: Variant;
  bFound: Boolean;

begin
  CheckBrowseMode;
  CursorPosChanged;
  AFields := TStringList.Create;
  SaveFields := TStringList.Create;
  try
    AFields.CommaText := StringReplace(KeyFields, ';', ',', [rfReplaceAll]);
    PartialLength := loPartialKey in Options;
    CaseInsensitive := loCaseInsensitive in Options;
    if VarIsArray(KeyValues) then
      Values := KeyValues
    else
      Values := VarArrayOf([KeyValues]);
    { save current record in case we cannot locate KeyValues }
    StartValues := VarArrayCreate([0, FieldCount], varVariant);
    for I := 0 to FieldCount -1 do
    begin
      StartValues[I] := Fields[I].Value;
      SaveFields.Add(Fields[I].FieldName);
    end;
    First;
    while not EOF do
    begin
      if CheckValues(AFields, Values, CaseInsensitive, PartialLength) then
        break;
      Next;
    end;
    { if not found, reset cursor to starting position }
    bFound := not EOF;
    if not bFound then
    begin
      First;
      while not EOF do
      begin
        if CheckValues(SaveFields, StartValues, False, False) then
          break;
        Next;
      end;
    end;
    Result := bFound;
  finally
    AFields.Free;
    SaveFields.Free;
  end;
end;

function TCustomSQLDataSet.Locate(const KeyFields: string; const KeyValues: Variant;
  Options: TLocateOptions): Boolean;
begin
  DoBeforeScroll;
  Result := LocateRecord(KeyFields, KeyValues, Options, True);
  if Result then
  begin
    Resync([rmExact, rmCenter]);
    DoAfterScroll;
  end;
end;

function TCustomSQLDataSet.Lookup(const KeyFields: string; const KeyValues: Variant;
  const ResultFields: string): Variant;
begin
  Result := Null;
  if LocateRecord(KeyFields, KeyValues, [], False) then
  begin
    SetTempState(dsCalcFields);
    try
      CalculateFields(Nil);
      Result := FieldValues[ResultFields];
    finally
      RestoreState(dsBrowse);
    end;
  end;
end;

procedure TCustomSQLDataSet.SetSchemaName(const Value: string);
begin
  if FSchemaName <> Value then
  begin
    PropertyChanged;
    FSchemaName := Value;
  end;
end;

procedure TCustomSQLDataSet.SetSchemaOption;
var
  Status: SQLResult;
  str: WideString;
  ASchemaName, ObjectName, CatalogName : WideString;
begin
  ObjectName := FSchemaInfo.ObjectName;
  if ObjectName <> '' then
  begin
    Status := GetInternalConnection.FISQLConnection.setStringOption(eConnQualifiedName, ObjectName);
    if Status <> 0 then
      SQLError(Status, exceptConnection);
    SetLength(Str, 256);
    Status := GetInternalConnection.FISQLConnection.getStringOption(eConnCatalogName, str);
    if Status <> 0 then
      SQLError(Status, exceptConnection);
    CatalogName := str;
    SetLength(Str, 256);
    Status := GetInternalConnection.FISQLConnection.getStringOption(eConnSchemaName, str);
    if Status <> 0 then
      SQLError(Status, exceptConnection);
    ASchemaName := str;
    SetLength(Str, 256);
    Status := GetInternalConnection.FISQLConnection.getStringOption(eConnObjectName, str);
    if Status <> 0 then
      SQLError(Status, exceptConnection);
    FSchemaInfo.ObjectName := str;
  end;
  if Length(CatalogName) = 0 then
    CataLogName := GetInternalConnection.FParams.Values[DATABASENAME_KEY];
  if Length(CatalogName) > 0 then
  begin
    Status := GetInternalConnection.FSQLMetaData.setStringOption(eMetaCatalogName, CatalogName);
    if Status <> 0 then
      SQLError(Status, exceptMetaData);
  end;
  (* by default, ASchemaName has been retrieved from getOption(eMetaSchemaName).
     if this is NOT set, then try TCustomDataSet.SchemaName;
     if this is NOT set, then try DefaultSchemaName;
     if this is NOT set, then try the UserName used to login;
     only if this is NOT set, get UserName from Parameter StringList *)
  if Length(ASchemaName) = 0 then
    ASchemaName := SchemaName;
  if Length(ASchemaName) = 0 then
  begin
    SetLength(Str, 256);
    Status := GetInternalConnection.FSQLMetadata.getStringOption(eMetaDefaultSchemaName, str);
    if Status = DBXERR_NONE then
        ASchemaName := str;
    if (Length(ASchemaName) <= 0) then
      ASchemaName := GetInternalConnection.GetLoginUsername;
    if (Length(ASchemaName) <= 0) then
      ASchemaName := GetInternalConnection.FParams.Values[szUSERNAME];
  end;
  if Length(ASchemaName) > 0 then
  begin
    Status := GetInternalConnection.FSQLMetaData.setStringOption(eMetaSchemaName, ASchemaName);
    if Status <> 0 then
      SQLError(Status, exceptMetaData);
  end;
end;
{ TSQLDataSet }

constructor TSQLDataSet.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCommandType := ctQuery;
  FGetMetadata := True;
end;

function TSQLDataSet.ExecSQL(ExecDirect: Boolean = False): Integer;
begin
  Result := inherited ExecSQL(ExecDirect);
end;

{ TSQLQuery }

constructor TSQLQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCommandType := ctQuery;
  FSQL := TWideStringList.Create;
  FGetMetaData := False;
  TWideStringList(SQL).OnChange := QueryChanged;
end;

destructor TSQLQuery.Destroy;
begin
  FSQL.Free;
  inherited Destroy;
end;

function TSQLQuery.ExecSQL(ExecDirect: Boolean = False): Integer;
begin
  Result := inherited ExecSQL(ExecDirect);
end;

procedure TSQLQuery.PrepareStatement;
var
  SQLText: Widestring;
  CurSection: TSqlToken;
  Value: WideString;
  Command: PWideChar;
begin
  if FCommandText = '' then
    SetSQL(SQL);
  if Length(CommandText) = 0 then
    DatabaseError(SEmptySQLStatement, Self);
  Command := PWideChar(CommandText);
  CurSection := stUnknown;
  CurSection := NextSQLToken(Command, Value, CurSection);
  if CurSection = stSelect then
    Inc(FSQLConnection.FSelectStatements);
  CheckStatement;
  SQLText := FNativeCommand;
  Check(FSQLCommand.prepare(PWideChar(SQLText), ParamCount), exceptCommand);
end;

procedure TSQLQuery.QueryChanged(Sender: TObject);
begin
  if not (csReading in ComponentState) then
  begin
    Close;
    SetPrepared(False);
    if ParamCheck or (csDesigning in ComponentState) then
    begin
      FCommandText := SQL.Text;
      FText := FCommandText;
      SetParamsFromSQL(nil, False);
    end
    else
      FText := SQL.Text;
    DataEvent(dePropertyChange, 0);
  end 
  else
    FText := FParams.ParseSQL(SQL.Text, False);
  SetFCommandText(FText);
end;

procedure TSQLQuery.SetSQL(Value: TWideStrings);
begin
  if SQL.Text <> Value.Text then
  begin
    Close;
    SQL.BeginUpdate;
    try
      SQL.Assign(Value);
    finally
      SQL.EndUpdate;
    end;
  end;
end;

{ TSQLStoredProc }

constructor TSQLStoredProc.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCommandType := ctStoredProc;
  FGetMetadata := True;
end;

function TSQLStoredProc.ExecProc: Integer;
begin
  Result := ExecSQL;
end;

procedure TSQLStoredProc.PrepareStatement;
var
  SQLText: Widestring;
begin
  if FCommandText = '' then
    SetStoredProcName(FStoredProcName);
  if Length(CommandText) = 0 then
    DatabaseError(SEmptySQLStatement, Self);
  CheckStatement;
  Check(FSQLCommand.SetOption(eCommStoredProc, Integer(True)), exceptCommand);
  Check(FSQLCommand.SetStringOption(eCommPackageName, FPackageName), exceptCommand);
  if FSchemaName <> '' then
    SQLText := AddQuoteCharToObjectName(Self, FSchemaName + '.' + FNativeCommand)
  else
    SQLText := AddQuoteCharToObjectName(Self, FNativeCommand);
  Check(FSQLCommand.prepare(PWideChar(SQLText), ParamCount), exceptCommand);
end;

procedure TSQLStoredProc.SetStoredProcName(Value: WideString);
begin
  //if FStoredProcName <> Value then
  //begin
    FStoredProcName := Value;
    SetCommandText(Value);
    if Assigned(FProcParams) then  // free output params if any
      FreeProcParams(FProcParams);
  //end;
end;

procedure TSQLStoredProc.SetPackageName(Value: WideString);
begin
  if FPackageName <> Value then
  begin
    FPackageName := Value;
    FSchemaInfo.PackageName := Value;
    if Assigned(FProcParams) then
      FreeProcParams(FProcParams);
    FStoredProcName := '';
    SetCommandText('');
  end;
end;

function TSQLStoredProc.NextRecordSet: TCustomSQLDataSet;
begin
  FGetNextRecordSet := True;
  SetState(dsInactive);
  CloseCursor;
  if Assigned(FieldDefs) then
    FieldDefs.Updated := False;
  try
    Active := True;
  finally
    FGetNextRecordSet := False;
  end;
  if Assigned(FSqlCursor ) then
    Result := TCustomSQLDataSet(Self)
  else
    Result := Nil;
end;

{ TSQLTable }

constructor TSQLTable.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCommandType := ctTable;
  FGetMetadata := True;
  FIndexFieldCount := -1;
  FMasterLink := TMasterDataLink.Create(Self);
  FIndexFields := TList.Create;
end;

destructor TSQLTable.Destroy;
begin
  FreeAndNil(FMasterLink);
  FreeAndNil(FIndexFields);
  inherited Destroy;
end;

procedure TSQLTable.DeleteRecords;
begin
  SQLConnection.ExecuteDirect('delete from ' + TableName);   { do not localize }
end;

function TSQLTable.GetIndexField(Index: Integer): TField;
begin
  if IndexName = '' then Result := nil
  else
  begin
    if FIndexFieldCount = -1 then
      RefreshIndexFields;
    Result := FIndexFields[Index];
  end;
end;

function TSQLTable.GetIndexFieldCount: Integer;
begin
  if IndexName = '' then Result := 0
  else if FIndexFieldCount >= 0 then Result := FIndexFieldCount
  else Result := RefreshIndexFields;
end;

procedure TSQLTable.GetIndexNames(List: TWideStrings);
begin
  FSQLConnection.GetIndexNames(FTableName,List);
end;

procedure TSQLTable.OpenCursor(InfoQuery: Boolean);
begin
  inherited OpenCursor(InfoQuery);
  if not FIsDetail and not FIndexDefsLoaded then
    AddIndexDefs(Self);
end;

procedure TSQLTable.AddParamsToQuery;
var
  I: Integer;
  Value: string;
begin
  if Pos('?', NativeCommand) = 0 then
  begin
    for I := 0 to Params.Count -1 do
    begin
      if Params[I].IsNull then
        Value := 'is NULL'
      else
        Value := '= ?';
      if I = 0 then
        NativeCommand := format('%s%s(%s %s)', [NativeCommand, SWhere, Params[I].Name, Value])
      else
        NativeCommand := format('%s%s(%s %s)', [NativeCommand, SAnd, Params[I].Name, Value]);
    end;
  end;
end;

procedure TSQLTable.SetDataSource(Value: TDataSource);
begin
  inherited SetDataSource(Value);
end;

function TSQLTable.GetQueryFromType: WideString;
begin
  if FNativeCommand <> '' then
    Result := FNativeCommand
  else
    Result := inherited GetQueryFromType;
end;

procedure TSQLTable.PrepareStatement;

  function GetFieldsForIndexName(IndexName: WideString): WideString;
  var
    DataSet: TCustomSQLDataSet;
    IdxName: WideString;
  begin
    DataSet := FSQLConnection.OpenSchemaTable(stIndexes, TableName,'','','');
    if not Assigned(DataSet) then FSQLConnection.SQLError(SQLResult(-1), exceptMetadata);
    try
      while not DataSet.EOF do
      begin
        IdxName := DataSet.FieldByName(IDX_NAME_FIELD).Value;
        if IdxName = IndexName then
        begin
          if Result = '' then
            Result := DataSet.FieldByName(COL_NAME_FIELD).Value
          else
            Result := Result + ';' + DataSet.FieldByName(COL_NAME_FIELD).Value;
        end;
        DataSet.Next;
      end;
    finally
      FSQLConnection.FreeSchemaTable(DataSet);
    end;
  end;

  function GetIndexFieldNames(FieldNames, IndexName: WideString): WideString;
  begin
    if (FieldNames = '') and (IndexName = '') then
      Result := ''
    else if FieldNames <> '' then
      Result := FieldNames
    else
      Result := GetFieldsForIndexName(IndexName);
  end;

var
  FDetailWhere, SQLText, IdxFieldNames: Widestring;
  FIndex, Pos1, Pos2: Integer;
  FName1, FName2, TempString1, TempString2: WideString;
  STableName : WideString;
begin  // first, convert TableName into valid Query.
  if Length(FTableName) = 0 then
    DatabaseError(SEmptySQLStatement, Self);
  if FNativeCommand = '' then  // otherwise, already prepared
  begin
    if (FDataLink.DataSource <> nil) and (MasterFields <> '') then
    begin
      FIsDetail := True;
      Pos1 := 1;
      Pos2 := 1;
      FIndex := 1;
      TempString1 := MasterFields;
      TempString2 := IndexFieldNames;
      while Pos1 <= Length(TempString1) do
        begin
          FName1 := ExtractFieldName(TempString1, Pos1);
          FName2 := ExtractFieldName(TempString2, Pos2);
          if FName1 = '' then Break;
          if FIndex = 1 then
            FDetailWhere := SWhere
          else
            FDetailWhere := FDetailWhere + SAnd;
          if FName2 = '' then
            FDetailWhere := FDetailWhere + FName1 + ' = :' + FName1
          else
            FDetailWhere := FDetailWhere + FName2 + ' = :' + FName1;
          Inc(FIndex);
        end;
      FCommandType := ctQuery;
      SetCommandText(SSelectStarFrom + AddQuoteCharToObjectName(Self ,FTableName)
                      + FDetailWhere);
    end else
    begin
      FIsDetail := False;
      IdxFieldNames := GetIndexFieldNames(IndexFieldNames, IndexName);
      if Self.FSchemaName <> '' then
        STableName := AddQuoteCharToObjectName(Self, FSchemaName + '.' + FTableName)
      else
        STableName := AddQuoteCharToObjectName(Self, FTableName);
      if IdxFieldNames = '' then
        FCommandText := SSelectStarFrom + STableName
      else
        FCommandText := SSelectStarFrom + STableName
                     + SOrderBy + StringReplace(IdxFieldNames, ';', ',', [rfReplaceAll]);
    end;
  end else if Params.Count > 0 then
    AddParamsToQuery;

  Inc(FSQLConnection.FSelectStatements);
  CheckStatement;
  SQLText := FNativeCommand;
  Check(FSQLCommand.prepare(PWideChar(SQLText), ParamCount), exceptCommand);
  FCommandType := ctTable;
  FCommandText := FTableName;
end;

function TSQLTable.RefreshIndexFields: Integer;
var
  I, Pos: Integer;
  Temp: WideString;
  FField: TField;
begin
  Result := 0;
  if not FIndexDefsLoaded then
    AddIndexDefs(Self);
  FIndexFields.Clear;
  for I := 0 to IndexDefs.Count - 1 do
  begin
    if WideCompareText(IndexDefs[I].Name, IndexName) = 0 then
    begin
      Temp := IndexDefs[I].Fields;
      Pos := 1;
      while Pos <= Length(Temp) do
      begin
        FField := FindField(ExtractFieldName(Temp, Pos));
        if FField = nil then
          Break
        else
          FIndexFields.Add(FField);
        Inc(Result);
      end;
      Break;
    end;
  end;
end;

function TSQLTable.GetMasterFields: WideString;
begin
  Result := FMasterLink.FieldNames;
end;

procedure TSQLTable.SetMasterFields(Value: WideString);
begin
  FMasterLink.FieldNames := Value;
  if not (csLoading in ComponentState) then
  begin
    Close;
    FreeStatement;
    FNativeCommand := '';
    FParams.clear;
  end;
end;

procedure TSQLTable.SetTableName(Value: WideString);
begin
  if FTableName <> Value then
  begin
    FNativeCommand := '';
    FTableName := Value;
    SetCommandText(Value);
  end;
end;

procedure TSQLTable.SetIndexFieldNames(Value: WideString);
begin
  if FIndexFieldNames <> Value then
  begin
    if (csDesigning in ComponentState) and not (csLoading in ComponentState) then
      if (TableName = '') and (Value <> '') then DatabaseError(SNoTableName,Self);
    FIndexFieldNames := Value;
    if FIndexFieldNames <> '' then
      SetIndexName('');
    FNativeCommand := '';
    SetPrepared(False);
  end;
end;

procedure TSQLTable.SetIndexField(Index: Integer; Value: TField);
begin
  GetIndexField(Index).Assign(Value);
end;

procedure TSQLTable.SetIndexName(Value: WideString);
begin
  if (csDesigning in ComponentState) and not (csLoading in ComponentState) then
    if (TableName = '') and (Value <> '') then DatabaseError(SNoTableName,Self);
  if FIndexName <> Value then
  begin
    FIndexName := Value;
    FNativeCommand := '';
    if Assigned(FSQLConnection) and (Value <> '') then
    begin
      SetIndexFieldNames('');  // clear out IndexFieldNames
      if (csDesigning in ComponentState) and not (csLoading in ComponentState) then
        AddIndexDefs(Self, Value);
    end;
    SetPrepared(False);
  end;
  FIndexFieldCount := -1;
end;

{$IFDEF MSWINDOWS}
procedure RegisterDbXpressLib(GetClassProc: Pointer);
begin
  GetDriver := GetClassProc;
  DllHandle := THandle(1);
end;
{$ENDIF}

end.
